# -*- coding: utf-8 -*-
#
#   Ming-Ke-Ming : Decentralized User Identity Authentication
#
#                                Written in 2019 by Moky <albert.moky@gmail.com>
#
# ==============================================================================
# MIT License
#
# Copyright (c) 2019 Albert Moky
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# ==============================================================================

from abc import ABC, abstractmethod
from typing import Optional

from .crypto import EncryptKey, SignKey
from .identifier import ID
from .meta import Meta
from .profile import Profile


class EntityDataSource(ABC):
    """This interface is for getting information for entity(user/group)

        Entity Data Source
        ~~~~~~~~~~~~~~~~~~

        1. meta for user, which is generated by the user's private key
        2. meta for group, which is generated by the founder's private key
        3. meta key, which can verify message sent by this user(or group founder)
        4. profile key, which can encrypt message for the receiver(user)
    """

    @abstractmethod
    def meta(self, identifier: ID) -> Optional[Meta]:
        """
        Get meta for entity ID

        :param identifier: entity ID
        :return:           meta info
        """
        raise NotImplemented

    @abstractmethod
    def profile(self, identifier: ID) -> Optional[Profile]:
        """
        Get profile for entity ID

        :param identifier: entity ID
        :return:           profile info
        """
        pass


class UserDataSource(EntityDataSource):
    """This interface is for getting information for user

        User Data Source
        ~~~~~~~~~~~~~~~~

        (Encryption/decryption)
        1. public key for encryption
           if profile.key not exists, means it is the same key with meta.key
        2. private keys for decryption
           the private keys paired with [profile.key, meta.key]

        (Signature/Verification)
        3. private key for signature
           the private key paired with meta.key
        4. public keys for verification
           [meta.key]
    """

    @abstractmethod
    def contacts(self, identifier: ID) -> Optional[list]:
        """
        Get user's contacts list

        :param identifier: user ID
        :return: contact ID list
        """
        pass

    @abstractmethod
    def public_key_for_encryption(self, identifier: ID) -> Optional[EncryptKey]:
        """
        Get user's public key for encryption
        (profile.key or meta.key)

        :param identifier: user ID
        :return: public key
        """
        pass

    @abstractmethod
    def private_keys_for_decryption(self, identifier: ID) -> Optional[list]:
        """
        Get user's private keys for decryption
        (which paired with [profile.key, meta.key])

        :param identifier: user ID
        :return: private keys
        """
        raise NotImplemented

    @abstractmethod
    def private_key_for_signature(self, identifier: ID) -> Optional[SignKey]:
        """
        Get user's private key for signature
        (which paired with profile.key or meta.key)

        :param identifier: user ID
        :return: private key
        """
        raise NotImplemented

    @abstractmethod
    def public_keys_for_verification(self, identifier: ID) -> Optional[list]:
        """
        Get user's public keys for verification
        [profile.key, meta.key]

        :param identifier: user ID
        :return: public keys
        """
        pass


class GroupDataSource(EntityDataSource):
    """This interface is for getting information for group

        Group Data Source
        ~~~~~~~~~~~~~~~~~

        1. founder has the same public key with the group's meta.key
        2. owner and members should be set complying with the consensus algorithm
    """

    @abstractmethod
    def founder(self, identifier: ID) -> Optional[ID]:
        """
        Get founder of the group

        :param identifier: group ID
        :return: founder ID
        """
        pass

    @abstractmethod
    def owner(self, identifier: ID) -> Optional[ID]:
        """
        Get current owner of the group

        :param identifier: group ID
        :return: owner ID
        """
        pass

    @abstractmethod
    def members(self, identifier: ID) -> Optional[list]:
        """
        Get all members in the group

        :param identifier: group ID
        :return: member ID list
        """
        pass
