"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnCodeDeployBlueGreenHook = exports.CfnTrafficRoutingType = void 0;
const cfn_hook_1 = require("./cfn-hook");
/**
 * The possible types of traffic shifting for the blue-green deployment configuration.
 *
 * The type of the {@link CfnTrafficRoutingConfig.type} property.
 */
var CfnTrafficRoutingType;
(function (CfnTrafficRoutingType) {
    CfnTrafficRoutingType["ALL_AT_ONCE"] = "AllAtOnce";
    CfnTrafficRoutingType["TIME_BASED_CANARY"] = "TimeBasedCanary";
    CfnTrafficRoutingType["TIME_BASED_LINEAR"] = "TimeBasedLinear";
})(CfnTrafficRoutingType = exports.CfnTrafficRoutingType || (exports.CfnTrafficRoutingType = {}));
/**
 * A CloudFormation Hook for CodeDeploy blue-green ECS deployments.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html#blue-green-template-reference
 */
class CfnCodeDeployBlueGreenHook extends cfn_hook_1.CfnHook {
    /**
     * Creates a new CodeDeploy blue-green ECS Hook.
     *
     * @param scope the scope to create the hook in (usually the containing Stack object).
     * @param id the identifier of the construct - will be used to generate the logical ID of the Hook.
     * @param props the properties of the Hook.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'AWS::CodeDeploy::BlueGreen',
        });
        this._serviceRole = props.serviceRole;
        this._applications = props.applications;
        this._trafficRoutingConfig = props.trafficRoutingConfig;
        this._additionalOptions = props.additionalOptions;
        this._lifecycleEventHooks = props.lifecycleEventHooks;
    }
    /**
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope, id, hookAttributes, options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        hookAttributes = hookAttributes || {};
        const hookProperties = options.parser.parseValue(hookAttributes.Properties);
        return new CfnCodeDeployBlueGreenHook(scope, id, {
            serviceRole: hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.ServiceRole,
            applications: (_a = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.Applications) === null || _a === void 0 ? void 0 : _a.map(applicationFromCloudFormation),
            trafficRoutingConfig: {
                type: (_b = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.Type,
                timeBasedCanary: {
                    stepPercentage: (_d = (_c = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _c === void 0 ? void 0 : _c.TimeBasedCanary) === null || _d === void 0 ? void 0 : _d.StepPercentage,
                    bakeTimeMins: (_f = (_e = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _e === void 0 ? void 0 : _e.TimeBasedCanary) === null || _f === void 0 ? void 0 : _f.BakeTimeMins,
                },
                timeBasedLinear: {
                    stepPercentage: (_h = (_g = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _g === void 0 ? void 0 : _g.TimeBasedLinear) === null || _h === void 0 ? void 0 : _h.StepPercentage,
                    bakeTimeMins: (_k = (_j = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _j === void 0 ? void 0 : _j.TimeBasedLinear) === null || _k === void 0 ? void 0 : _k.BakeTimeMins,
                },
            },
            additionalOptions: {
                terminationWaitTimeInMinutes: (_l = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.AdditionalOptions) === null || _l === void 0 ? void 0 : _l.TerminationWaitTimeInMinutes,
            },
            lifecycleEventHooks: {
                beforeInstall: (_m = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.BeforeInstall,
                afterInstall: (_o = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.AfterInstall,
                afterAllowTestTraffic: (_p = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.AfterAllowTestTraffic,
                beforeAllowTraffic: (_q = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.BeforeAllowTraffic,
                afterAllowTraffic: (_r = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _r === void 0 ? void 0 : _r.AfterAllowTraffic,
            },
        });
        function applicationFromCloudFormation(app) {
            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w;
            const target = findResource((_a = app === null || app === void 0 ? void 0 : app.Target) === null || _a === void 0 ? void 0 : _a.LogicalID);
            const taskDefinitions = (_c = (_b = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _b === void 0 ? void 0 : _b.TaskDefinitions) === null || _c === void 0 ? void 0 : _c.map((td) => findResource(td));
            const taskSets = (_e = (_d = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _d === void 0 ? void 0 : _d.TaskSets) === null || _e === void 0 ? void 0 : _e.map((ts) => findResource(ts));
            const prodTrafficRoute = findResource((_h = (_g = (_f = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _f === void 0 ? void 0 : _f.TrafficRouting) === null || _g === void 0 ? void 0 : _g.ProdTrafficRoute) === null || _h === void 0 ? void 0 : _h.LogicalID);
            const testTrafficRoute = findResource((_l = (_k = (_j = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _j === void 0 ? void 0 : _j.TrafficRouting) === null || _k === void 0 ? void 0 : _k.TestTrafficRoute) === null || _l === void 0 ? void 0 : _l.LogicalID);
            const targetGroups = (_p = (_o = (_m = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _m === void 0 ? void 0 : _m.TrafficRouting) === null || _o === void 0 ? void 0 : _o.TargetGroups) === null || _p === void 0 ? void 0 : _p.map((tg) => findResource(tg));
            return {
                target: {
                    type: (_q = app === null || app === void 0 ? void 0 : app.Target) === null || _q === void 0 ? void 0 : _q.Type,
                    logicalId: target === null || target === void 0 ? void 0 : target.logicalId,
                },
                ecsAttributes: {
                    taskDefinitions: taskDefinitions === null || taskDefinitions === void 0 ? void 0 : taskDefinitions.map(td => td === null || td === void 0 ? void 0 : td.logicalId),
                    taskSets: taskSets === null || taskSets === void 0 ? void 0 : taskSets.map(ts => ts === null || ts === void 0 ? void 0 : ts.logicalId),
                    trafficRouting: {
                        prodTrafficRoute: {
                            type: (_t = (_s = (_r = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _r === void 0 ? void 0 : _r.TrafficRouting) === null || _s === void 0 ? void 0 : _s.ProdTrafficRoute) === null || _t === void 0 ? void 0 : _t.Type,
                            logicalId: prodTrafficRoute === null || prodTrafficRoute === void 0 ? void 0 : prodTrafficRoute.logicalId,
                        },
                        testTrafficRoute: {
                            type: (_w = (_v = (_u = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _u === void 0 ? void 0 : _u.TrafficRouting) === null || _v === void 0 ? void 0 : _v.TestTrafficRoute) === null || _w === void 0 ? void 0 : _w.Type,
                            logicalId: testTrafficRoute === null || testTrafficRoute === void 0 ? void 0 : testTrafficRoute.logicalId,
                        },
                        targetGroups: targetGroups === null || targetGroups === void 0 ? void 0 : targetGroups.map((tg) => tg === null || tg === void 0 ? void 0 : tg.logicalId),
                    },
                },
            };
        }
        function findResource(logicalId) {
            if (logicalId == null) {
                return undefined;
            }
            const ret = options.parser.finder.findResource(logicalId);
            if (!ret) {
                throw new Error(`Hook '${id}' references resource '${logicalId}' that was not found in the template`);
            }
            return ret;
        }
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    get serviceRole() {
        return this._serviceRole;
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    set serviceRole(serviceRole) {
        this._serviceRole = serviceRole;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    get applications() {
        return this._applications;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    set applications(value) {
        this._applications = value;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    get trafficRoutingConfig() {
        return this._trafficRoutingConfig;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    set trafficRoutingConfig(value) {
        this._trafficRoutingConfig = value;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    get additionalOptions() {
        return this._additionalOptions;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    set additionalOptions(value) {
        this._additionalOptions = value;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     *
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    get lifecycleEventHooks() {
        return this._lifecycleEventHooks;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     *
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    set lifecycleEventHooks(value) {
        this._lifecycleEventHooks = value;
    }
    /**
     *
     */
    renderProperties(_props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        return {
            ServiceRole: this.serviceRole,
            Applications: this.applications.map((app) => ({
                Target: {
                    Type: app.target.type,
                    LogicalID: app.target.logicalId,
                },
                ECSAttributes: {
                    TaskDefinitions: app.ecsAttributes.taskDefinitions,
                    TaskSets: app.ecsAttributes.taskSets,
                    TrafficRouting: {
                        ProdTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.prodTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.prodTrafficRoute.logicalId,
                        },
                        TestTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.testTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.testTrafficRoute.logicalId,
                        },
                        TargetGroups: app.ecsAttributes.trafficRouting.targetGroups,
                    },
                },
            })),
            TrafficRoutingConfig: {
                Type: (_a = this.trafficRoutingConfig) === null || _a === void 0 ? void 0 : _a.type,
                TimeBasedCanary: {
                    StepPercentage: (_c = (_b = this.trafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.timeBasedCanary) === null || _c === void 0 ? void 0 : _c.stepPercentage,
                    BakeTimeMins: (_e = (_d = this.trafficRoutingConfig) === null || _d === void 0 ? void 0 : _d.timeBasedCanary) === null || _e === void 0 ? void 0 : _e.bakeTimeMins,
                },
                TimeBasedLinear: {
                    StepPercentage: (_g = (_f = this.trafficRoutingConfig) === null || _f === void 0 ? void 0 : _f.timeBasedLinear) === null || _g === void 0 ? void 0 : _g.stepPercentage,
                    BakeTimeMins: (_j = (_h = this.trafficRoutingConfig) === null || _h === void 0 ? void 0 : _h.timeBasedLinear) === null || _j === void 0 ? void 0 : _j.bakeTimeMins,
                },
            },
            AdditionalOptions: {
                TerminationWaitTimeInMinutes: (_k = this.additionalOptions) === null || _k === void 0 ? void 0 : _k.terminationWaitTimeInMinutes,
            },
            LifecycleEventHooks: {
                BeforeInstall: (_l = this.lifecycleEventHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall,
                AfterInstall: (_m = this.lifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.afterInstall,
                AfterAllowTestTraffic: (_o = this.lifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.afterAllowTestTraffic,
                BeforeAllowTraffic: (_p = this.lifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.beforeAllowTraffic,
                AfterAllowTraffic: (_q = this.lifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.afterAllowTraffic,
            },
        };
    }
}
exports.CfnCodeDeployBlueGreenHook = CfnCodeDeployBlueGreenHook;
//# sourceMappingURL=data:application/json;base64,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