"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnJson = void 0;
const construct_compat_1 = require("./construct-compat");
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The creation stack of this resolvable which will be appended to errors thrown during resolution.
         *
         * If this returns an empty array the stack will not be attached.
         */
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    /**
     * Produce the Token's value at resolution time.
     */
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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