"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value.
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    /**
     *
     */
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    /**
     *
     */
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    /**
     *
     */
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    /**
     *
     */
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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