"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoveTag = exports.Tags = exports.Tag = void 0;
// import * as cxapi from '@aws-cdk/cx-api';
const aspect_1 = require("./aspect");
const tag_manager_1 = require("./tag-manager");
const annotations_1 = require("./annotations");
/**
 * The common functionality for Tag and Remove Tag Aspects
 */
class TagBase {
    constructor(key, props = {}) {
        this.key = key;
        this.props = props;
    }
    /**
     * All aspects can visit an IConstruct.
     */
    visit(construct) {
        if (tag_manager_1.TagManager.isTaggable(construct)) {
            this.applyTag(construct);
        }
    }
}
/**
 * The Tag Aspect will handle adding a tag to this node and cascading tags to children.
 */
class Tag extends TagBase {
    /**
     *
     */
    constructor(key, value, props = {}) {
        super(key, props);
        this.defaultPriority = 100;
        if (value === undefined) {
            throw new Error('Tag must have a value');
        }
        this.value = value;
    }
    /**
     * (deprecated) DEPRECATED: add tags to the node of a construct and all its the taggable children.
     *
     * @deprecated use `Tags.of(scope).add()`
     */
    static add(scope, key, value, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.add(scope,k,v)', 'Use "Tags.of(scope).add(k,v)" instead');
        Tags.of(scope).add(key, value, props);
    }
    /**
     * (deprecated) DEPRECATED: remove tags to the node of a construct and all its the taggable children.
     *
     * @deprecated use `Tags.of(scope).remove()`
     */
    static remove(scope, key, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.remove(scope,k,v)', 'Use "Tags.of(scope).remove(k,v)" instead');
        Tags.of(scope).remove(key, props);
    }
    /**
     *
     */
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.setTag(this.key, this.value, this.props.priority !== undefined ? this.props.priority : this.defaultPriority, this.props.applyToLaunchedInstances !== false);
        }
    }
}
exports.Tag = Tag;
/**
 * Manages AWS tags for all resources within a construct scope.
 */
class Tags {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the tags API for this scope.
     *
     * @param scope The scope.
     */
    static of(scope) {
        return new Tags(scope);
    }
    /**
     * add tags to the node of a construct and all its the taggable children.
     */
    add(key, value, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new Tag(key, value, props));
    }
    /**
     * remove tags to the node of a construct and all its the taggable children.
     */
    remove(key, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new RemoveTag(key, props));
    }
}
exports.Tags = Tags;
/**
 * The RemoveTag Aspect will handle removing tags from this node and children.
 */
class RemoveTag extends TagBase {
    /**
     *
     */
    constructor(key, props = {}) {
        super(key, props);
        this.defaultPriority = 200;
    }
    /**
     *
     */
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.removeTag(this.key, this.props.priority !== undefined ? this.props.priority : this.defaultPriority);
        }
    }
}
exports.RemoveTag = RemoveTag;
//# sourceMappingURL=data:application/json;base64,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