"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultStackSynthesizer = exports.BOOTSTRAP_QUALIFIER_CONTEXT = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const assets_1 = require("../assets");
const cfn_fn_1 = require("../cfn-fn");
const cfn_parameter_1 = require("../cfn-parameter");
const cfn_rule_1 = require("../cfn-rule");
const token_1 = require("../token");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
exports.BOOTSTRAP_QUALIFIER_CONTEXT = '@aws-cdk/core:bootstrapQualifier';
/* eslint-disable max-len */
/**
 * The minimum bootstrap stack version required by this app.
 */
const MIN_BOOTSTRAP_STACK_VERSION = 6;
/**
 * Uses conventionally named roles and reify asset storage locations.
 *
 * This synthesizer is the only StackSynthesizer that generates
 * an asset manifest, and is required to deploy CDK applications using the
 * `@aws-cdk/app-delivery` CI/CD library.
 *
 * Requires the environment to have been bootstrapped with Bootstrap Stack V2.
 *
 * @stability stable
 */
class DefaultStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super();
        this.props = props;
        this.files = {};
        this.dockerImages = {};
        for (const key in props) {
            if (props.hasOwnProperty(key)) {
                validateNoToken(key);
            }
        }
        function validateNoToken(key) {
            const prop = props[key];
            if (typeof prop === 'string' && token_1.Token.isUnresolved(prop)) {
                throw new Error(`DefaultSynthesizer property '${key}' cannot contain tokens; only the following placeholder strings are allowed: ` + [
                    '${Qualifier}',
                    cxapi.EnvironmentPlaceholders.CURRENT_REGION,
                    cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT,
                    cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
                ].join(', '));
            }
        }
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (this._stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._stack = stack;
        const qualifier = (_c = (_b = this.props.qualifier) !== null && _b !== void 0 ? _b : stack.node.tryGetContext(exports.BOOTSTRAP_QUALIFIER_CONTEXT)) !== null && _c !== void 0 ? _c : DefaultStackSynthesizer.DEFAULT_QUALIFIER;
        this.qualifier = qualifier;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${Qualifier}: always
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            s = replaceAll(s, '${Qualifier}', qualifier);
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        /* eslint-disable max-len */
        this.bucketName = specialize((_d = this.props.fileAssetsBucketName) !== null && _d !== void 0 ? _d : DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME);
        this.repositoryName = specialize((_e = this.props.imageAssetsRepositoryName) !== null && _e !== void 0 ? _e : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME);
        this._deployRoleArn = specialize((_f = this.props.deployRoleArn) !== null && _f !== void 0 ? _f : DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN);
        this._cloudFormationExecutionRoleArn = specialize((_g = this.props.cloudFormationExecutionRole) !== null && _g !== void 0 ? _g : DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN);
        this.fileAssetPublishingRoleArn = specialize((_h = this.props.fileAssetPublishingRoleArn) !== null && _h !== void 0 ? _h : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN);
        this.imageAssetPublishingRoleArn = specialize((_j = this.props.imageAssetPublishingRoleArn) !== null && _j !== void 0 ? _j : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN);
        this.lookupRoleArn = specialize((_k = this.props.lookupRoleArn) !== null && _k !== void 0 ? _k : DefaultStackSynthesizer.DEFAULT_LOOKUP_ROLE_ARN);
        this.bucketPrefix = specialize((_l = this.props.bucketPrefix) !== null && _l !== void 0 ? _l : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX);
        this.dockerTagPrefix = specialize((_m = this.props.dockerTagPrefix) !== null && _m !== void 0 ? _m : DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX);
        this.bootstrapStackVersionSsmParameter = replaceAll((_o = this.props.bootstrapStackVersionSsmParameter) !== null && _o !== void 0 ? _o : DefaultStackSynthesizer.DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER, '${Qualifier}', qualifier);
        /* eslint-enable max-len */
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.bucketName);
        validateFileAssetSource(asset);
        const extension = asset.fileName != undefined ? path.extname(asset.fileName) : '';
        const objectKey = this.bucketPrefix + asset.sourceHash + (asset.packaging === assets_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : extension);
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                executable: asset.executable,
                packaging: asset.packaging,
            },
            destinations: {
                [this.manifestEnvName]: {
                    bucketName: this.bucketName,
                    objectKey,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.fileAssetPublishingExternalId,
                },
            },
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${this.bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${this.bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(this.bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
            s3Url: httpUrl,
        };
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.repositoryName);
        validateDockerImageAssetSource(asset);
        const imageTag = this.dockerTagPrefix + asset.sourceHash;
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                executable: asset.executable,
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations: {
                [this.manifestEnvName]: {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.imageAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.imageAssetPublishingExternalId,
                },
            },
        };
        const { account, region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        // Return CFN expression
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${region}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    /**
     * Have the stack write out its template.
     *
     * @stability stable
     */
    synthesizeStackTemplate(stack, session) {
        stack._synthesizeTemplate(session, this.lookupRoleArn);
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        var _b;
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.qualifier);
        // Must be done here -- if it's done in bind() (called in the Stack's constructor)
        // then it will become impossible to set context after that.
        //
        // If it's done AFTER _synthesizeTemplate(), then the template won't contain the
        // right constructs.
        if ((_b = this.props.generateBootstrapVersionRule) !== null && _b !== void 0 ? _b : true) {
            addBootstrapVersionRule(this.stack, MIN_BOOTSTRAP_STACK_VERSION, this.bootstrapStackVersionSsmParameter);
        }
        this.synthesizeStackTemplate(this.stack, session);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        this.emitStackArtifact(this.stack, session, {
            assumeRoleExternalId: this.props.deployRoleExternalId,
            assumeRoleArn: this._deployRoleArn,
            cloudFormationExecutionRoleArn: this._cloudFormationExecutionRoleArn,
            stackTemplateAssetObjectUrl: templateManifestUrl,
            requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
            bootstrapStackVersionSsmParameter: this.bootstrapStackVersionSsmParameter,
            additionalDependencies: [artifactId],
        });
    }
    /**
     * Returns the ARN of the deploy Role.
     *
     * @stability stable
     */
    get deployRoleArn() {
        if (!this._deployRoleArn) {
            throw new Error('deployRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._deployRoleArn;
    }
    /**
     * Returns the ARN of the CFN execution Role.
     *
     * @stability stable
     */
    get cloudFormationExecutionRoleArn() {
        if (!this._cloudFormationExecutionRoleArn) {
            throw new Error('cloudFormationExecutionRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._cloudFormationExecutionRoleArn;
    }
    /**
     * @stability stable
     */
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const templatePath = path.join(session.assembly.outdir, this.stack.templateFile);
        const template = fs.readFileSync(templatePath, { encoding: 'utf-8' });
        const sourceHash = _shared_1.contentHash(template);
        this.addFileAsset({
            fileName: this.stack.templateFile,
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash,
        });
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        //
        // ALSO: it would be great to reuse the return value of `addFileAsset()` here, except those contain
        // CloudFormation REFERENCES to locations, not actual locations (can contain `{ Ref: AWS::Region }` and
        // `{ Ref: SomeParameter }` etc). We therefore have to duplicate some logic here :(.
        const extension = path.extname(this.stack.templateFile);
        return `s3://${this.bucketName}/${this.bucketPrefix}${sourceHash}${extension}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        fs.writeFileSync(outPath, JSON.stringify(manifest, undefined, 2));
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
                requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
                bootstrapStackVersionSsmParameter: this.bootstrapStackVersionSsmParameter,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        _shared_1.assertBound(this.stack);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.DefaultStackSynthesizer = DefaultStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
DefaultStackSynthesizer[_a] = { fqn: "@aws-cdk/core.DefaultStackSynthesizer", version: "1.131.0" };
/**
 * Default ARN qualifier.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_QUALIFIER = 'hnb659fds';
/**
 * Default CloudFormation role ARN.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-cfn-exec-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default deploy role ARN.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-deploy-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for file (S3) assets.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-file-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for image (ECR) assets.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-image-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default lookup role ARN for missing values.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_LOOKUP_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-lookup-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default image assets repository name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME = 'cdk-${Qualifier}-container-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Default file assets bucket name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME = 'cdk-${Qualifier}-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Name of the CloudFormation Export with the asset key name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME = 'CdkBootstrap-${Qualifier}-FileAssetKeyArn';
/**
 * Default file asset prefix.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX = '';
/**
 * Default Docker asset prefix.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX = '';
/**
 * Default bootstrap stack version SSM parameter.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER = '/cdk-bootstrap/${Qualifier}/version';
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return token_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? cfn_fn_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
/**
 * Add a CfnRule to the Stack which checks the current version of the bootstrap stack this template is targeting
 *
 * The CLI normally checks this, but in a pipeline the CLI is not involved
 * so we encode this rule into the template in a way that CloudFormation will check it.
 */
function addBootstrapVersionRule(stack, requiredVersion, bootstrapStackVersionSsmParameter) {
    // Because of https://github.com/aws/aws-cdk/blob/master/packages/assert-internal/lib/synth-utils.ts#L74
    // synthesize() may be called more than once on a stack in unit tests, and the below would break
    // if we execute it a second time. Guard against the constructs already existing.
    if (stack.node.tryFindChild('BootstrapVersion')) {
        return;
    }
    const param = new cfn_parameter_1.CfnParameter(stack, 'BootstrapVersion', {
        type: 'AWS::SSM::Parameter::Value<String>',
        description: `Version of the CDK Bootstrap resources in this environment, automatically retrieved from SSM Parameter Store. ${cxapi.SSMPARAM_NO_INVALIDATE}`,
        default: bootstrapStackVersionSsmParameter,
    });
    // There is no >= check in CloudFormation, so we have to check the number
    // is NOT in [1, 2, 3, ... <required> - 1]
    const oldVersions = range(1, requiredVersion).map(n => `${n}`);
    new cfn_rule_1.CfnRule(stack, 'CheckBootstrapVersion', {
        assertions: [
            {
                assert: cfn_fn_1.Fn.conditionNot(cfn_fn_1.Fn.conditionContains(oldVersions, param.valueAsString)),
                assertDescription: `CDK bootstrap stack version ${requiredVersion} required. Please run 'cdk bootstrap' with a recent version of the CDK CLI.`,
            },
        ],
    });
}
function range(startIncl, endExcl) {
    const ret = new Array();
    for (let i = startIncl; i < endExcl; i++) {
        ret.push(i);
    }
    return ret;
}
function validateFileAssetSource(asset) {
    if (!!asset.executable === !!asset.fileName) {
        throw new Error(`Exactly one of 'fileName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    if (!!asset.packaging !== !!asset.fileName) {
        throw new Error(`'packaging' is expected in combination with 'fileName', got: ${JSON.stringify(asset)}`);
    }
}
function validateDockerImageAssetSource(asset) {
    if (!!asset.executable === !!asset.directoryName) {
        throw new Error(`Exactly one of 'directoryName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    check('dockerBuildArgs');
    check('dockerBuildTarget');
    check('dockerFile');
    function check(key) {
        if (asset[key] && !asset.directoryName) {
            throw new Error(`'${key}' is only allowed in combination with 'directoryName', got: ${JSON.stringify(asset)}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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