"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the original deployment environment.
 *
 * This deployment environment is restricted in cross-environment deployments,
 * CI/CD deployments, and will use up CloudFormation parameters in your template.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 *
 * @stability stable
 */
class LegacyStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    constructor() {
        super(...arguments);
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        if (this.stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this.stack = stack;
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addFileAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addDockerImageAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        _shared_1.assertBound(this.stack);
        this.synthesizeStackTemplate(this.stack, session);
        // Just do the default stuff, nothing special
        this.emitStackArtifact(this.stack, session);
    }
    doAddDockerImageAsset(asset) {
        var _b, _c;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_c = (_b = asset.repositoryName) !== null && _b !== void 0 ? _b : repositoryNameOverride) !== null && _c !== void 0 ? _c : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            if (!asset.directoryName) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            if (!asset.fileName || !asset.packaging) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
LegacyStackSynthesizer[_a] = { fqn: "@aws-cdk/core.LegacyStackSynthesizer", version: "1.131.0" };
//# sourceMappingURL=data:application/json;base64,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