# pylint: skip-file
# fmt: off
r"""The unpolarized, space-like anomalous dimension :math:`\gamma_{qg}^{(3)}`."""
import numba as nb
import numpy as np

from .....harmonics import cache as c
from .....harmonics.log_functions import lm11, lm12, lm13, lm14, lm15


@nb.njit(cache=True)
def gamma_qg_nf3(n, cache):
    r"""Return the part proportional to :math:`nf^3` of :math:`\gamma_{qg}^{(3)}`.

    The expression is copied exact from :eqref:`3.12` of :cite:`Davies:2016jie`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{qg}^{(3)}|_{nf^3}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    Sm2 = c.get(c.Sm2, cache, n, is_singlet=True)
    S3 = c.get(c.S3, cache, n)
    S21 = c.get(c.S21, cache, n)
    Sm3 = c.get(c.Sm3, cache, n, is_singlet=True)
    S4 = c.get(c.S4, cache, n)
    Sm4 = c.get(c.Sm4, cache, n, is_singlet=True)
    S31 = c.get(c.S31, cache, n)
    S211 = c.get(c.S211, cache, n)
    return 1.3333333333333333 * (
        44.56685134331718 / (-1.0 + n)
        - 82.37037037037037 / np.power(n, 5)
        + 95.30864197530865 / np.power(n, 4)
        - 298.6951686088834 / np.power(n, 3)
        + 334.4519003852186 / np.power(n, 2)
        - 576.1641081960868 / n
        - 156.44444444444446 / np.power(1.0 + n, 6)
        + 271.4074074074074 / np.power(1.0 + n, 5)
        - 142.6172839506173 / np.power(1.0 + n, 4)
        + 49.20926725891911 / np.power(1.0 + n, 3)
        + 242.3162373306182 / np.power(1.0 + n, 2)
        + 383.95514040473176 / (1.0 + n)
        + 75.85185185185185 / np.power(2.0 + n, 5)
        - 18.962962962962962 / np.power(2.0 + n, 4)
        - 28.97119341563786 / np.power(2.0 + n, 3)
        + 57.904384241653375 / np.power(2.0 + n, 2)
        + 153.56036440484917 / (2.0 + n)
        - (7.725651577503429 * S1) / (-1.0 + n)
        + (35.55555555555556 * S1) / np.power(n, 5)
        - (53.333333333333336 * S1) / np.power(n, 4)
        + (149.7283950617284 * S1) / np.power(n, 3)
        - (189.49794238683128 * S1) / np.power(n, 2)
        + (219.77429764880566 * S1) / n
        - (71.11111111111111 * S1) / np.power(1.0 + n, 5)
        + (75.85185185185185 * S1) / np.power(1.0 + n, 4)
        + (45.4320987654321 * S1) / np.power(1.0 + n, 3)
        - (24.691358024691358 * S1) / np.power(1.0 + n, 2)
        - (242.01773110008048 * S1) / (1.0 + n)
        + (37.925925925925924 * S1) / np.power(2.0 + n, 4)
        + (53.72839506172839 * S1) / np.power(2.0 + n, 3)
        + (39.76954732510288 * S1) / np.power(2.0 + n, 2)
        + (33.84214810968268 * S1) / (2.0 + n)
        - (8.954732510288066 * S2) / (-1.0 + n)
        + (28.444444444444443 * S2) / np.power(n, 4)
        - (47.407407407407405 * S2) / np.power(n, 3)
        + (117.33333333333333 * S2) / np.power(n, 2)
        - (128.52674897119343 * S2) / n
        - (71.11111111111111 * S2) / np.power(1.0 + n, 4)
        + (9.481481481481481 * S2) / np.power(1.0 + n, 3)
        + (60.44444444444444 * S2) / np.power(1.0 + n, 2)
        - (11.588477366255145 * S2) / (1.0 + n)
        + (56.888888888888886 * S2) / np.power(2.0 + n, 3)
        + (112.19753086419753 * S2) / np.power(2.0 + n, 2)
        + (144.98765432098764 * S2) / (2.0 + n)
        - (2.3703703703703702 * (np.power(S1, 2) + S2)) / (-1.0 + n)
        + (2.3703703703703702 * (np.power(S1, 2) + S2)) / np.power(n, 3)
        + (3.3580246913580245 * (np.power(S1, 2) + S2)) / np.power(n, 2)
        - (13.695473251028806 * (np.power(S1, 2) + S2)) / n
        + (7.111111111111111 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 4)
        + (9.481481481481481 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 3)
        - (2.962962962962963 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 2)
        + (53.76131687242798 * (np.power(S1, 2) + S2)) / (1.0 + n)
        - (9.481481481481481 * (np.power(S1, 2) + S2)) / np.power(2.0 + n, 3)
        - (21.333333333333332 * (np.power(S1, 2) + S2)) / np.power(2.0 + n, 2)
        - (38.650205761316876 * (np.power(S1, 2) + S2)) / (2.0 + n)
        - (3.1604938271604937 * S21) / (-1.0 + n)
        + (7.111111111111111 * S21) / np.power(n, 3)
        - (10.666666666666666 * S21) / np.power(n, 2)
        + (32.0 * S21) / n
        - (14.222222222222221 * S21) / np.power(1.0 + n, 3)
        - (53.333333333333336 * S21) / (1.0 + n)
        + (9.481481481481481 * S21) / np.power(2.0 + n, 2)
        + (24.493827160493826 * S21) / (2.0 + n)
        - (3.1604938271604937 * S3) / (-1.0 + n)
        + (7.111111111111111 * S3) / np.power(n, 3)
        - (10.666666666666666 * S3) / np.power(n, 2)
        + (28.049382716049383 * S3) / n
        - (14.222222222222221 * S3) / np.power(1.0 + n, 3)
        - (43.06172839506173 * S3) / (1.0 + n)
        + (9.481481481481481 * S3) / np.power(2.0 + n, 2)
        + (14.222222222222221 * S3) / (2.0 + n)
        - (3.1604938271604937 * (S1 * S2 - 1.0 * S21 + S3)) / (-1.0 + n)
        + (7.111111111111111 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(n, 3)
        - (10.666666666666666 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(n, 2)
        + (32.0 * (S1 * S2 - 1.0 * S21 + S3)) / n
        - (14.222222222222221 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(1.0 + n, 3)
        - (53.333333333333336 * (S1 * S2 - 1.0 * S21 + S3)) / (1.0 + n)
        + (9.481481481481481 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(2.0 + n, 2)
        + (24.493827160493826 * (S1 * S2 - 1.0 * S21 + S3)) / (2.0 + n)
        + (0.5267489711934157 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / (-1.0 + n)
        - (1.1851851851851851 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(n, 3)
        + (1.9753086419753085 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(n, 2)
        - (4.674897119341564 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / n
        + (2.3703703703703702 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(1.0 + n, 3)
        + (7.57201646090535 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / (1.0 + n)
        - (1.5802469135802468 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(2.0 + n, 2)
        - (2.765432098765432 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / (2.0 + n)
        - (
            1.1851851851851851
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / n
        + (
            2.3703703703703702
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / (1.0 + n)
        - (
            2.3703703703703702
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / (2.0 + n)
        + (3.5555555555555554 * S4) / n
        - (7.111111111111111 * S4) / (1.0 + n)
        + (7.111111111111111 * S4) / (2.0 + n)
    ) + 3.0 * (
        2.5381463063368415 / (-1.0 + n)
        + 3.5555555555555554 / np.power(n, 5)
        + 5.728395061728395 / np.power(n, 4)
        + 4.559670781893004 / np.power(n, 3)
        + 2.036401939671693 / np.power(n, 2)
        - 1.1169664019346115 / n
        + 26.074074074074073 / np.power(1.0 + n, 5)
        - 56.098765432098766 / np.power(1.0 + n, 4)
        + 70.0246913580247 / np.power(1.0 + n, 3)
        - 56.58563233464838 / np.power(1.0 + n, 2)
        + 23.268759906664844 / (1.0 + n)
        + 18.962962962962962 / np.power(2.0 + n, 5)
        - 37.925925925925924 / np.power(2.0 + n, 4)
        + 22.386831275720166 / np.power(2.0 + n, 3)
        + 25.899964373170548 / np.power(2.0 + n, 2)
        - 33.71051594275432 / (2.0 + n)
        + (0.3511659807956104 * S1) / (-1.0 + n)
        - (9.481481481481481 * S1) / np.power(n, 4)
        + (8.938271604938272 * S1) / np.power(n, 3)
        - (17.432098765432098 * S1) / np.power(n, 2)
        + (15.015825807984449 * S1) / n
        + (14.222222222222221 * S1) / np.power(1.0 + n, 4)
        - (1.7777777777777777 * S1) / np.power(1.0 + n, 3)
        - (14.534979423868313 * S1) / np.power(1.0 + n, 2)
        + (48.28933603835209 * S1) / (1.0 + n)
        + (9.481481481481481 * S1) / np.power(2.0 + n, 4)
        - (46.617283950617285 * S1) / np.power(2.0 + n, 3)
        - (42.27160493827161 * S1) / np.power(2.0 + n, 2)
        - (64.96148967346869 * S1) / (2.0 + n)
        + (1.7119341563786008 * S2) / (-1.0 + n)
        - (2.3703703703703702 * S2) / np.power(n, 3)
        + (0.7407407407407407 * S2) / np.power(n, 2)
        - (7.703703703703703 * S2) / n
        + (9.481481481481481 * S2) / np.power(1.0 + n, 3)
        - (15.012345679012345 * S2) / np.power(1.0 + n, 2)
        + (27.308641975308642 * S2) / (1.0 + n)
        - (4.7407407407407405 * S2) / np.power(2.0 + n, 3)
        + (2.765432098765432 * S2) / np.power(2.0 + n, 2)
        - (21.020576131687243 * S2) / (2.0 + n)
        + (0.8559670781893004 * (np.power(S1, 2) + S2)) / (-1.0 + n)
        - (1.1851851851851851 * (np.power(S1, 2) + S2)) / np.power(n, 3)
        + (0.37037037037037035 * (np.power(S1, 2) + S2)) / np.power(n, 2)
        - (2.60082304526749 * (np.power(S1, 2) + S2)) / n
        + (2.9135802469135803 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 2)
        + (5.275720164609053 * (np.power(S1, 2) + S2)) / (1.0 + n)
        + (2.3703703703703702 * (np.power(S1, 2) + S2)) / np.power(2.0 + n, 3)
        - (6.518518518518518 * (np.power(S1, 2) + S2)) / np.power(2.0 + n, 2)
        - (2.8724279835390947 * (np.power(S1, 2) + S2)) / (2.0 + n)
        - (3.950617283950617 * S21) / n
        - (2.3703703703703702 * S21) / np.power(1.0 + n, 2)
        + (7.901234567901234 * S21) / (1.0 + n)
        + (2.3703703703703702 * S21) / np.power(2.0 + n, 2)
        - (7.901234567901234 * S21) / (2.0 + n)
        + (1.1851851851851851 * S211) / n
        - (2.3703703703703702 * S211) / (1.0 + n)
        + (2.3703703703703702 * S211) / (2.0 + n)
        - (3.950617283950617 * S3) / n
        + (2.3703703703703702 * S3) / np.power(1.0 + n, 2)
        + (3.1604938271604937 * S3) / (1.0 + n)
        - (2.3703703703703702 * S3) / np.power(2.0 + n, 2)
        - (3.1604938271604937 * S3) / (2.0 + n)
        + (3.950617283950617 * (S1 * S2 - 1.0 * S21 + S3)) / n
        + (2.3703703703703702 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(1.0 + n, 2)
        - (7.901234567901234 * (S1 * S2 - 1.0 * S21 + S3)) / (1.0 + n)
        - (2.3703703703703702 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(2.0 + n, 2)
        + (7.901234567901234 * (S1 * S2 - 1.0 * S21 + S3)) / (2.0 + n)
        - (0.6584362139917695 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / n
        - (0.3950617283950617 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(1.0 + n, 2)
        + (1.316872427983539 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / (1.0 + n)
        + (0.3950617283950617 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(2.0 + n, 2)
        - (1.316872427983539 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / (2.0 + n)
        + (1.1851851851851851 * S31) / n
        - (2.3703703703703702 * S31) / (1.0 + n)
        + (2.3703703703703702 * S31) / (2.0 + n)
        + (
            1.1851851851851851
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / n
        - (
            2.3703703703703702
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / (1.0 + n)
        + (
            2.3703703703703702
            * (
                0.041666666666666664 * np.power(S1, 4)
                + 0.25 * np.power(S1, 2) * S2
                + 0.125 * np.power(S2, 2)
                + 0.3333333333333333 * S1 * S3
                + 0.25 * S4
            )
        )
        / (2.0 + n)
        + (3.5555555555555554 * S4) / n
        - (7.111111111111111 * S4) / (1.0 + n)
        + (7.111111111111111 * S4) / (2.0 + n)
        - (0.5925925925925926 * (np.power(S2, 2) + S4)) / n
        + (1.1851851851851851 * (np.power(S2, 2) + S4)) / (1.0 + n)
        - (1.1851851851851851 * (np.power(S2, 2) + S4)) / (2.0 + n)
        - (1.1851851851851851 * (S1 * S3 - 1.0 * S31 + S4)) / n
        + (2.3703703703703702 * (S1 * S3 - 1.0 * S31 + S4)) / (1.0 + n)
        - (2.3703703703703702 * (S1 * S3 - 1.0 * S31 + S4)) / (2.0 + n)
        + (
            1.1851851851851851
            * (S1 * S21 - 2.0 * S211 + S31 + 0.5 * (np.power(S2, 2) + S4))
        )
        / n
        - (
            2.3703703703703702
            * (S1 * S21 - 2.0 * S211 + S31 + 0.5 * (np.power(S2, 2) + S4))
        )
        / (1.0 + n)
        + (
            2.3703703703703702
            * (S1 * S21 - 2.0 * S211 + S31 + 0.5 * (np.power(S2, 2) + S4))
        )
        / (2.0 + n)
        - (
            1.1851851851851851
            * (
                S211
                + 0.5 * (S1 * S3 + S1 * (S1 * S2 - 2.0 * S21 + S3) - 2.0 * S31 + S4)
            )
        )
        / n
        + (
            2.3703703703703702
            * (
                S211
                + 0.5 * (S1 * S3 + S1 * (S1 * S2 - 2.0 * S21 + S3) - 2.0 * S31 + S4)
            )
        )
        / (1.0 + n)
        - (
            2.3703703703703702
            * (
                S211
                + 0.5 * (S1 * S3 + S1 * (S1 * S2 - 2.0 * S21 + S3) - 2.0 * S31 + S4)
            )
        )
        / (2.0 + n)
        + (2.5020576131687244 * Sm2) / n
        + (0.5925925925925926 * Sm2) / np.power(1.0 + n, 2)
        - (0.6584362139917695 * Sm2) / (1.0 + n)
        + (1.8436213991769548 * Sm2) / (2.0 + n)
        - (7.901234567901234 * Sm3) / n
        + (11.061728395061728 * Sm3) / (1.0 + n)
        - (11.061728395061728 * Sm3) / (2.0 + n)
        + (4.7407407407407405 * Sm4) / n
        - (9.481481481481481 * Sm4) / (1.0 + n)
        + (9.481481481481481 * Sm4) / (2.0 + n)
    )


@nb.njit(cache=True)
def gamma_qg_nf1(n, cache, variation):
    r"""Return the part proportional to :math:`nf^1` of :math:`\gamma_{qg}^{(3)}`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{qg}^{(3)}|_{nf^1}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3 = c.get(c.S3, cache, n)
    S4 = c.get(c.S4, cache, n)
    S5 = c.get(c.S5, cache, n)
    S1m2 = ((1 - 2 * n)/((-1 + n) * n) + S1)/n
    common = -7871.5226542038545/np.power(-1. + n,3) + 41639.78057369606/np.power(-1. + n,2) + 14103.703703703704/np.power(n,7) + 2588.8395061728397/np.power(n,6) + 68802.34242841466/np.power(n,5) - 35.68779444531073*lm14(n,S1,S2,S3,S4) - 1.8518518518518519*lm15(n,S1,S2,S3,S4,S5)
    if variation == 1:
        fit = -3829.861172460453/(-1. + n) + 277661.01447681/((-1. + n)*np.power(n,4)) - 153705.0394973762/((-1. + n)*np.power(n,3)) - 123955.9749794338/((-1. + n)*np.power(n,2)) + (335.2491670956783*np.power(S1,3))/n
    elif variation == 2:
        fit = -16578.93780360809/(-1. + n) + 371184.7701572534/((-1. + n)*np.power(n,3)) - 474716.0631314892/((-1. + n)*np.power(n,2)) + 103531.29297423577/((-1. + n)*n) + (581.7910219196752*np.power(S1,3))/n
    elif variation == 3:
        fit = 73188.49465256406/(-1. + n) - 1.3878834240687524e6/((-1. + n)*np.power(n,3)) + 1.3878834240687524e6/((-1. + n)*np.power(n,2)) - 561325.2602274221/((-1. + n)*n) - (1258.5072756524442*np.power(S1,3))/n
    elif variation == 4:
        fit = 115739.6851234958/(-1. + n) - 800561.2000440155/np.power(n,3) - 161259.43874202168/(1. + n) + (1168.102310595733*np.power(S1,3))/n
    elif variation == 5:
        fit = -577377.77555826/np.power(n,3) + 218313.25893291482/((-1. + n)*(2. + n)) - (40574.52923613038*n)/((-1. + n)*(2. + n)) + (1048.1382811275018*np.power(S1,3))/n
    elif variation == 6:
        fit = -24286.423116959464/(-1. + n) - 259529.64966439368/np.power(n,3) + 174613.1308005301/np.power(1. + n,2) + (719.2485137514881*np.power(S1,3))/n
    elif variation == 7:
        fit = -6656.017275123271/(-1. + n) - 373193.64052889764/np.power(n,3) + 445077.5302307679/np.power(1. + n,3) + (419.81089748690897*np.power(S1,3))/n
    elif variation == 8:
        fit = 57972.59678227429/(-1. + n) - 568070.7064955179/np.power(n,3) + (1870.9228018163747*np.power(S1,3))/n + 34977.83890482135*lm11(n,S1)
    elif variation == 9:
        fit = 41478.104162566255/(-1. + n) - 501323.7102676334/np.power(n,3) + (3164.5089720058513*np.power(S1,3))/n - 11580.107285479033*lm12(n,S1,S2)
    elif variation == 10:
        fit = 51616.27557607398/(-1. + n) - 544576.0709777217/np.power(n,3) + (7439.948419246559*np.power(S1,3))/n + 5726.50989183176*lm13(n,S1,S2,S3)
    elif variation == 11:
        fit = 28806.15019908585/(-1. + n) - 451566.26083838515/np.power(n,3) - (12473.783867993085*np.power(S1,2))/n + (3294.9337937707714*np.power(S1,3))/n
    elif variation == 12:
        fit = 37482.972438867444/(-1. + n) - 611475.0619976323/np.power(n,3) + (1682.353677404603*np.power(S1,3))/n - 27422.092055164114*S1m2
    elif variation == 13:
        fit = -20169.736525290085/(-1. + n) - 359487.2089232927/np.power(n,4) - (1758.492741848954*np.power(S1,3))/n - 1687.5920186043782*lm13(n,S1,S2,S3)
    elif variation == 14:
        fit = -162566.79581253263/(-1. + n) + 325164.3218798365/np.power(n,2) - (14099.70433097991*np.power(S1,3))/n - 12258.938297046334*lm13(n,S1,S2,S3)
    elif variation == 15:
        fit = 57695.42721524939/(-1. + n) - 158184.06830196152/((-1. + n)*n) + (4988.683159589739*np.power(S1,3))/(-1. + n) - (4988.683159589739*np.power(S1,3))/((-1. + n)*n) + 4112.752778007585*lm13(n,S1,S2,S3)
    elif variation == 16:
        fit = -84798.1894219582/(-1. + n) + 343059.113935714/(1. + n) + (20782.509856638342*np.power(S1,3))/n + 17908.937319094526*lm13(n,S1,S2,S3)
    elif variation == 17:
        fit = -93394.46772816063/(-1. + n) + 333595.2518693284/np.power(1. + n,2) - (5399.828105194443*np.power(S1,3))/n - 5213.884458465402*lm13(n,S1,S2,S3)
    elif variation == 18:
        fit = -133546.79046670574/(-1. + n) + 1.4142556623732254e6/np.power(1. + n,3) - (14866.88256285107*np.power(S1,3))/n - 12469.755904736707*lm13(n,S1,S2,S3)
    elif variation == 19:
        fit = -95715.2472187035/(-1. + n) + (136522.95069090303*np.power(S1,3))/n - 810742.2678533106*lm11(n,S1) + 138459.79936753682*lm13(n,S1,S2,S3)
    elif variation == 20:
        fit = 166665.22586748275/(-1. + n) + (54308.34321851309*np.power(S1,3))/n + 223223.32402508403*S1m2 + 52341.86551625599*lm13(n,S1,S2,S3)
    else:
        fit = -544.8767261921181/(-1. + n) - 17974.360446164636/np.power(n,4) + 13883.0507238405/((-1. + n)*np.power(n,4)) - 234383.7038186229/np.power(n,3) - 58520.18467044376/((-1. + n)*np.power(n,3)) + 16258.216093991825/np.power(n,2) + 39460.569297891474/((-1. + n)*np.power(n,2)) - 30798.901777757397/((-1. + n)*n) + 92966.65963019966/np.power(1. + n,3) + 25410.419133492927/np.power(1. + n,2) + 9089.983759684617/(1. + n) + 10915.662946645742/((-1. + n)*(2. + n)) - (2028.7264618065192*n)/((-1. + n)*(2. + n)) - (623.6891933996543*np.power(S1,2))/n + (249.43415797948697*np.power(S1,3))/(-1. + n) + (9797.76983028744*np.power(S1,3))/n - (249.43415797948697*np.power(S1,3))/((-1. + n)*n) + 9790.061598495997*S1m2 - 38788.22144742446*lm11(n,S1) - 579.0053642739516*lm12(n,S1,S2) + 9345.984709693692*lm13(n,S1,S2,S3)
    return common + fit


@nb.njit(cache=True)
def gamma_qg_nf2(n, cache, variation):
    r"""Return the part proportional to :math:`nf^2` of :math:`\gamma_{qg}^{(3)}`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{qg}^{(3)}|_{nf^2}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3 = c.get(c.S3, cache, n)
    S4 = c.get(c.S4, cache, n)
    S5 = c.get(c.S5, cache, n)
    S1m2 = ((1 - 2 * n)/((-1 + n) * n) + S1)/n
    common = 175.35644829905792/np.power(-1. + n,2) - 1991.111111111111/np.power(n,7) + 2069.3333333333335/np.power(n,6) - 7229.376633440217/np.power(n,5) + 3.511659807956104*lm14(n,S1,S2,S3,S4) + 0.411522633744856*lm15(n,S1,S2,S3,S4,S5)
    if variation == 1:
        fit = -187.58666786190324/(-1. + n) + 2506.6691233283627/((-1. + n)*np.power(n,4)) - 8027.943939980643/((-1. + n)*np.power(n,3)) + 5521.274816652281/((-1. + n)*np.power(n,2)) - (58.990476994719124*np.power(S1,3))/n
    elif variation == 2:
        fit = -302.6828285615236/(-1. + n) - 3289.3417941585412/((-1. + n)*np.power(n,3)) + 2354.681608299279/((-1. + n)*np.power(n,2)) + 934.6601858592619/((-1. + n)*n) - (56.76474553486762*np.power(S1,3))/n
    elif variation == 3:
        fit = 507.7199269075272/(-1. + n) - 19169.864079952302/((-1. + n)*np.power(n,3)) + 19169.864079952302/((-1. + n)*np.power(n,2)) - 5067.534239934793/((-1. + n)*n) - (73.3785972451255*np.power(S1,3))/n
    elif variation == 4:
        fit = -586.9843120112163/np.power(n,3) + 2347.682271699974/(-1. + np.power(n,2)) - (563.9548315608548*n)/(-1. + np.power(n,2)) - (51.471642335801754*np.power(S1,3))/n
    elif variation == 5:
        fit = 1427.871824890965/np.power(n,3) + 1664.8663722017177/((-1. + n)*(2. + n)) - (519.3103167713406*n)/((-1. + n)*(2. + n)) - (52.55465404036084*np.power(S1,3))/n
    elif variation == 6:
        fit = -372.2644926993197/(-1. + n) + 4297.342299841074/np.power(n,3) + 1576.3730616993998/np.power(1. + n,2) - (55.523806286719065*np.power(S1,3))/n
    elif variation == 7:
        fit = -213.100650275311/(-1. + n) + 3271.2061068606395/np.power(n,3) + 4018.072557355197/np.power(1. + n,3) - (58.22707029541625*np.power(S1,3))/n
    elif variation == 8:
        fit = 370.3537883913313/(-1. + n) + 1511.8942170593587/np.power(n,3) - (45.1267169313305*np.power(S1,3))/n + 315.77306215881396*lm11(n,S1)
    elif variation == 9:
        fit = 221.44474751391837/(-1. + n) + 2114.4729940426164/np.power(n,3) - (33.44847496679631*np.power(S1,3))/n - 104.5429349598605*lm12(n,S1,S2)
    elif variation == 10:
        fit = 312.97016852823697/(-1. + n) + 1723.999173627645/np.power(n,3) + (5.149352932277194*np.power(S1,3))/n + 51.69780697278038*lm13(n,S1,S2,S3)
    elif variation == 11:
        fit = 107.04483511591005/(-1. + n) + 2563.6734782188273/np.power(n,3) - (112.61087168425806*np.power(S1,2))/n - (32.27102528323112*np.power(S1,3))/n
    elif variation == 12:
        fit = 185.37748270798173/(-1. + n) + 1120.0482175043628/np.power(n,3) - (46.8290819632629*np.power(S1,3))/n - 247.56126307924774*S1m2
    elif variation == 13:
        fit = 540.2277219469585/(-1. + n) + 1138.0515673425464/np.power(n,4) + (34.26944433093038*np.power(S1,3))/n + 75.16910189000741*lm13(n,S1,S2,S3)
    elif variation == 14:
        fit = 991.0231674711375/(-1. + n) - 1029.3934164377968/np.power(n,2) + (73.33880436179972*np.power(S1,3))/n + 108.63548651363628*lm13(n,S1,S2,S3)
    elif variation == 15:
        fit = 293.7250107608868/(-1. + n) + 500.7733860652629/((-1. + n)*n) + (12.909479583459072*np.power(S1,3))/(-1. + n) - (12.909479583459072*np.power(S1,3))/((-1. + n)*n) + 56.806580712459464*lm13(n,S1,S2,S3)
    elif variation == 16:
        fit = 1830.8702341223307/(-1. + np.power(n,2)) - (341.21796603948076*n)/(-1. + np.power(n,2)) - (37.090044129076624*np.power(S1,3))/n + 13.131120082421122*lm13(n,S1,S2,S3)
    elif variation == 17:
        fit = 772.039920114616/(-1. + n) - 1056.0837488071672/np.power(1. + n,2) + (45.79705175230928*np.power(S1,3))/n + 86.33251059684723*lm13(n,S1,S2,S3)
    elif variation == 18:
        fit = 899.1526871082096/(-1. + n) - 4477.199280629624/np.power(1. + n,3) + (75.76750938256234*np.power(S1,3))/n + 109.30288524966886*lm13(n,S1,S2,S3)
    elif variation == 19:
        fit = 779.3869596506777/(-1. + n) - (403.496933795239*np.power(S1,3))/n + 2566.618465799714*lm11(n,S1) - 368.50442658872794*lm13(n,S1,S2,S3)
    elif variation == 20:
        fit = -51.24762545788382/(-1. + n) - (143.22491139800567*np.power(S1,3))/n - 706.672302847881*S1m2 - 95.87539511556255*lm13(n,S1,S2,S3)
    else:
        fit = 242.67920756807254/(-1. + n) + 56.90257836712732/np.power(n,4) + 125.33345616641815/((-1. + n)*np.power(n,4)) + 872.1762000017136/np.power(n,3) - 1524.3574907045743/((-1. + n)*np.power(n,3)) - 51.46967082188984/np.power(n,2) + 1352.2910252451932/((-1. + n)*np.power(n,2)) - 181.60503340051338/((-1. + n)*n) - 22.956336163721335/np.power(1. + n,3) + 26.014465644611633/np.power(1. + n,2) + 83.2433186100859/((-1. + n)*(2. + n)) - (25.96551583856703*n)/((-1. + n)*(2. + n)) + 208.92762529111525/(-1. + np.power(n,2)) - (45.25863988001678*n)/(-1. + np.power(n,2)) - (5.630543584212903*np.power(S1,2))/n + (0.6454739791729537*np.power(S1,3))/(-1. + n) - (45.70380092200368*np.power(S1,3))/n - (0.6454739791729537*np.power(S1,3))/((-1. + n)*n) - 47.71167829635644*S1m2 + 144.1195763979264*lm11(n,S1) - 5.227146747993025*lm12(n,S1,S2) + 1.8347835156765138*lm13(n,S1,S2,S3)
    return common + fit

@nb.njit(cache=True)
def gamma_qg(n, nf, cache, variation):
    r"""Compute the |N3LO| quark-gluon singlet anomalous dimension.

    Parameters
    ----------
    n : complex
        Mellin moment
    nf : int
        Number of active flavors
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| quark-gluon singlet anomalous dimension
        :math:`\gamma_{qg}^{(3)}(N)`

    """
    return (
        +nf * gamma_qg_nf1(n, cache, variation)
        + nf**2 * gamma_qg_nf2(n, cache, variation)
        + nf**3 * gamma_qg_nf3(n, cache)
    )
