"""
Blueprints represent the system's structure per SKU.
Initially (at import time from component_blueprints.*) each blueprint is a mapping between SKUs and component IDs.
get_blueprints_by_sku converts the blueprint local variables (ends with "BLUEPRINT") from mappings to actual blueprints.
On importing the blueprints, they've already been adjusted with the blueprint corresponding to the proper SKU.
"""

from infiniguard_health.blueprints.component_blueprints.eth_blueprints import ETH_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.fc_blueprints import FC_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.ddes_blueprints import DDES_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.idracs_blueprints import IDRACS_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.volumes_blueprints import VOLUMES_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.dimms_blueprints import DIMMS_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.fans_blueprints import FANS_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.psus_blueprints import PSUS_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.services_blueprints import SERVICES_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.temps_blueprints import TEMPS_BLUEPRINT
from infiniguard_health.blueprints.component_blueprints.policies_blueprints import POLICIES_BLUEPRINT
from infiniguard_health.infiniguard_sysconfig import get_sku_from_iguard_sysconfig
from infiniguard_health.logger_config import is_test
from logging import getLogger

_logger = getLogger(__name__)


def get_blueprints_by_sku(blueprints, sku):
    """Converts the blueprints from dicts of SKU mappings
       to the actual blueprints which correspond to the sku provided.
       If SKU mapping does not contain the given SKU, it fallbacks to the default blueprint."""
    blueprints_by_sku_or_default = {}
    for blueprint_var_name, blueprint in blueprints.items():
        blueprint_key = sku if sku in blueprint else 'default'
        _logger.info(f'{blueprint_var_name} is generated by "{blueprint_key}"')
        blueprints_by_sku_or_default[blueprint_var_name] = blueprint[blueprint_key]
    return blueprints_by_sku_or_default


if is_test():
    SKU = 'NOD-91121'
else:
    SKU = get_sku_from_iguard_sysconfig()


blueprints = {var_name: var for var_name, var in locals().items() if var_name.endswith('BLUEPRINT')}
locals().update(get_blueprints_by_sku(blueprints, SKU))

# Intentionally below SKU resolution process.
from infiniguard_health.blueprints.component_blueprints.dynamic_blueprints import BONDS_BLUEPRINT, SNAPSHOTS_BLUEPRINT
