from cursed_geobr.utils import select_metadata, download_gpkg


def read_disaster_risk_area(year=2010, simplified=True, verbose=False):
    """ Download official data of disaster risk areas as an sf object.
    
     This function reads the the official data of disaster risk areas in Brazil. It specifically focuses
 on geodynamic and hydro-meteorological disasters capable of triggering landslides and floods. The
 data set covers the whole country. Each risk area polygon (known as 'BATER') has unique code id (column 'geo_bater').
 The data set brings information on the extent to which the risk area polygons overlap with census
 tracts and block faces (column "acuracia") and number of ris areas within each risk area (column 'num').
 Orignal data were generated by IBGE and CEMADEN. For more information about the methodology, see deails
 at https://www.ibge.gov.br/geociencias/organizacao-do-territorio/tipologias-do-territorio/21538-populacao-em-areas-de-risco-no-brasil.html

    Parameters
    ----------
    year : int, optional
        Year of the data, by default 2010
    simplified: boolean, by default True
        Data 'type', indicating whether the function returns the 'original' dataset 
        with high resolution or a dataset with 'simplified' borders (Default)
    verbose : bool, optional
        by default False
    
    Returns
    -------
    gpd.GeoDataFrame
        Metadata and geopackage of selected states
    
    Raises
    ------
    Exception
        If parameters are not found or not well defined

    Example
    -------
    >>> from cursed_geobr import read_disaster_risk_area

    # Read specific state at a given year
    >>> df = read_disaster_risk_area(year=2010)
    """

    metadata = select_metadata("disaster_risk_area", year=year, simplified=simplified)

    gdf = download_gpkg(metadata)

    return gdf
