# Copyright (c) gocept gmbh & co. kg
# See also LICENSE.txt

from __future__ import unicode_literals, print_function
from gocept.net.configure.zones import (
    Zone, ForwardZone, RR, Zones, ReverseZone, NodeAddr,
    MixedRecordTypesException)
from netaddr import ip
import configobj
import datetime
import mock
import os
import os.path as p
import pkg_resources
import pytz
import re
import shutil
import tempfile
import unittest

LIST_NODES = [
    {'name': 'vm00',
     'location': 'whq',
     'parameters': {
         'interfaces': {
             'fe': {
                 'networks': {
                     '195.62.125.0/24': ['195.62.125.10'],
                     '2a02:248:101:62::/64': ['2a02:248:101:62::5b']
                 }},
             'srv': {
                 'networks': {
                     '212.122.41.128/25': ['212.122.41.136'],
                     '172.22.48.0/20': ['172.22.48.20', '172.22.48.33'],
                     '2a02:248:101:63::/64': ['2a02:248:101:63::5b']
                 }}},
         'reverses': {'195.62.125.10': 'www.example.com.'}
     }}
]


class ZoneTest(unittest.TestCase):

    def test_forward_ipv4(self):
        z = ForwardZone('gocept.net')
        z.add_a('vm00.fe.whq', ip.IPAddress('195.62.125.10'))
        self.assertIn(RR.A('vm00.fe.whq',
                           ip.IPAddress('195.62.125.10')), z.records)

    def test_forward_ipv6(self):
        z = ForwardZone('gocept.net')
        z.add_a('vm00.srv.whq', ip.IPAddress('2a02:248:101:63::5b'))
        self.assertIn(RR.AAAA('vm00.srv.whq',
                              ip.IPAddress('2a02:248:101:63::5b')),
                      z.records)

    def test_cannot_mix_a_and_cname(self):
        z = ForwardZone('gocept.net')
        z.add_a('vm00', ip.IPAddress('2a02:248:101:63::5b'))
        with self.assertRaises(MixedRecordTypesException):
            z.add_cname('vm00', 'other')

    def test_first_cname_wins(self):
        z = ForwardZone('gocept.net')
        z.add_cname('vm00', 'first')
        z.add_cname('vm00', 'second')
        self.assertEqual([RR.CNAME('vm00', 'first')], z.records)

    def test_render_forward(self):
        config = configobj.ConfigObj(pkg_resources.resource_stream(
            __name__, 'fixtures/configure-zones.cfg'))
        z = ForwardZone('gocept.net', parent_zones=Zones(config))
        z.add_a('vm00', ip.IPAddress('212.122.41.136'))
        z.add_a('vm00', ip.IPAddress('2a02:248:101:63::5b'))
        z.add_cname('vm00.srv.whq', 'vm00')
        self.maxDiff = 1024
        self.assertEqual(z.render(2014021900), """\
; generated by configure-zones
$TTL 86400
$ORIGIN gocept.net.
@               86400   IN      SOA ns1.gocept.net. hostmaster.gocept.net. (
                                        2014021900 ; serial
                                        10800 ; refresh
                                        900 ; retry
                                        2419200 ; expire
                                        1800 ; neg ttl
                                )
                                NS      ns1.gocept.net.
                                NS      ns2.gocept.net.
$TTL 7200
vm00                            A       212.122.41.136
vm00                            AAAA    2a02:248:101:63::5b
vm00.srv.whq                    CNAME   vm00
""")


class ReverseZoneTest(unittest.TestCase):

    def test_reverse_zone_is_private(self):
        r = ReverseZone('22.172.in-addr.arpa', ip.IPNetwork('172.22.0.0/16'))
        self.assertTrue(r.private)

    def test_reverse_zone_is_public(self):
        r = ReverseZone('125.62.195.in-addr.arpa',
                        ip.IPNetwork('195.62.125.0/24'))
        self.assertFalse(r.private)

    def test_add_ptr_strips_prefix(self):
        z = ReverseZone('1.0.1.0.8.4.2.0.2.0.a.2.ip6.arpa',
                        ip.IPNetwork('2a02:238:101::/48'))
        z.add_ptr(ip.IPAddress('2a02:248:101:63::5b'), 'vm00')
        self.assertEqual(
            RR.PTR('b.5.0.0.0.0.0.0.0.0.0.0.0.0.0.0.3.6.0.0', 'vm00'),
            z.records[0])

    def test_add_pre_strips_correct_prefix_for_cidr_zones(self):
        z = ReverseZone('128-25.41.122.212.in-addr.arpa',
                        prefix=ip.IPNetwork('212.122.41.128/25'),
                        strip_suffix='41.122.212.in-addr.arpa')
        z.add_ptr(ip.IPAddress('212.122.41.152'), 'rt.gocept.com.')
        self.assertEqual(RR.PTR('152', 'rt.gocept.com.'), z.records[0])


class ZoneSaveTest(unittest.TestCase):

    def setUp(self):
        self.pridir = tempfile.mkdtemp('.pri')
        self.zones = mock.Mock('Zones')
        self.zones.pridir = self.pridir
        self.zones.nameservers = ['ns1.local', 'ns2.local']
        self.zones.suffix = 'gocept.net'
        self.zones.ttl = 7200
        self.z = Zone('gocept.net-intern', 'gocept.net',
                      parent_zones=self.zones)
        self.filename = p.join(self.pridir, 'gocept.net-intern.zone')

    def tearDown(self):
        shutil.rmtree(self.zones.pridir)

    def test_fullpath(self):
        self.assertEqual(self.filename, self.z.fullpath())

    @mock.patch('gocept.net.utils.now')
    def test_serial_default_to_zero_unless_zone_exists(self, now):
        now.return_value = datetime.datetime(2014, 2, 19, tzinfo=pytz.utc)
        self.assertIs(self.z.parse_serial(), None)

    def test_parse_serial(self):
        with open(self.filename, 'w') as f:
            f.write(self.z.render(2013122205))
        self.assertEqual(2013122205, self.z.parse_serial())

    @mock.patch('gocept.net.utils.now')
    def test_save_returns_false_if_unchanged(self, now):
        now.return_value = datetime.datetime(2014, 2, 17, tzinfo=pytz.utc)
        with open(self.filename, 'w') as f:
            f.write(self.z.render(2014021703))
        self.assertFalse(self.z.save())
        with open(self.filename) as f:
            self.assertEqual(f.read(), self.z.render(2014021703))

    @mock.patch('gocept.net.utils.now')
    def test_save_returns_true_if_changed(self, now):
        now.return_value = datetime.datetime(2014, 2, 17, tzinfo=pytz.utc)
        with open(self.filename, 'w') as f:
            f.write(self.z.render(2014021703))
        self.z.records.append(RR.A('vm01', ip.IPAddress('192.168.1.1')))
        self.assertTrue(self.z.save())
        with open(self.filename) as f:
            self.assertEqual(f.read(), self.z.render(2014021704))

    def test_include_static_snippets(self):
        inc = [tempfile.NamedTemporaryFile(prefix='inc1.', dir=self.pridir),
               tempfile.NamedTemporaryFile(prefix='inc2.', dir=self.pridir)]
        for i, f in enumerate(inc):
            f.write('; include {}\n'.format(i))
            f.flush()
        zone = self.z = Zone(
            'gocept.net-intern', 'gocept.net', include=[f.name for f in inc],
            parent_zones=self.zones)
        self.assertRegexpMatches(zone.render(0), re.compile(
            r'; include 0\n.*; include 1\n', re.S))
        for f in inc:
            f.close()


class ZonesTest(unittest.TestCase):

    def setUp(self):
        self.pridir = tempfile.mkdtemp('.pri')
        self.c = configobj.ConfigObj(pkg_resources.resource_stream(
            __name__, 'fixtures/configure-zones.cfg'))
        self.c['settings']['pridir'] = self.pridir
        self.c['external']['include'] = []
        self.c['internal']['include'] = []
        self.z = Zones(self.c)

    def tearDown(self):
        shutil.rmtree(self.c['settings']['pridir'])

    def test_init_external_forward_zone(self):
        self.assertEqual(self.z.external_forward.name, 'gocept.net-external')
        self.assertEqual(self.z.external_forward.origin, 'gocept.net')

    def test_init_internal_forward_zone(self):
        self.assertEqual(self.z.internal_forward.name, 'gocept.net-internal')
        self.assertEqual(self.z.internal_forward.origin, 'gocept.net')

    def test_create_reverses_should_use_explicit_name(self):
        r = self.z.create_reverse_zones({
            '212.122.41.128/25': '128-25.41.122.212.in-addr.arpa.'})
        z = r[ip.IPNetwork('212.122.41.128/25')]
        self.assertEqual(z.name, '128-25.41.122.212.in-addr.arpa')
        self.assertEqual(z.origin, '128-25.41.122.212.in-addr.arpa')
        self.assertEqual(z.prefix, ip.IPNetwork('212.122.41.128/25'))

    def test_create_implicit_ipv4_reverse_zone_name(self):
        r = self.z.create_reverse_zones({'195.62.125.0/24': ''})
        zone = r[ip.IPNetwork('195.62.125.0/24')]
        self.assertEqual(zone.name, '125.62.195.in-addr.arpa')
        self.assertEqual(zone.origin, '125.62.195.in-addr.arpa')
        self.assertEqual(zone.prefix, ip.IPNetwork('195.62.125.0/24'))

    def test_default_ipv4_reverse_name(self):
        self.assertEqual(
            '16.172.in-addr.arpa.',
            self.z.default_reverse_name(ip.IPNetwork('172.16.0.0/16')))

    def test_default_ipv6_reverse_name(self):
        self.assertEqual(
            '1.0.1.0.8.4.2.0.2.0.a.2.ip6.arpa.',
            self.z.default_reverse_name(ip.IPNetwork('2a02:248:101::/48')))

    def test_add_addr_creates_public_records(self):
        self.z.add_addr('vm00', ip.IPAddress('195.62.125.33'), ['vm00.ipv4'])
        self.assertIn(RR.A('vm00', ip.IPAddress('195.62.125.33')),
                      self.z.external_forward.records)
        self.assertIn(RR.A('vm00', ip.IPAddress('195.62.125.33')),
                      self.z.internal_forward.records)
        self.assertIn(RR.CNAME('vm00.ipv4', 'vm00'),
                      self.z.external_forward.records)
        self.assertIn(RR.CNAME('vm00.ipv4', 'vm00'),
                      self.z.internal_forward.records)

    def test_add_addr_creates_no_public_records(self):
        self.z.add_addr('vm00', ip.IPAddress('172.22.48.20'), ['vm00.ipv4'])
        self.assertNotIn(RR.A('vm00', ip.IPAddress('172.22.48.20')),
                         self.z.external_forward.records)
        self.assertNotIn(RR.CNAME('vm00.ipv4', 'vm00'),
                         self.z.external_forward.records)

    def test_update_creates_all_zone_files(self):
        self.assertTrue(self.z.update_zones())
        self.assertEqual(sorted(os.listdir(self.pridir)), [
            '1.0.1.0.8.4.2.0.2.0.a.2.ip6.arpa.zone',
            '125.62.195.in-addr.arpa.zone',
            '128-25.41.122.212.in-addr.arpa.zone',
            '22.172.in-addr.arpa.zone',
            'gocept.net-external.zone',
            'gocept.net-internal.zone'])

    def test_list_internal_zones(self):
        self.assertEqual(sorted(z.name for z in self.z.all_internal_zones()), [
            '1.0.1.0.8.4.2.0.2.0.a.2.ip6.arpa',
            '125.62.195.in-addr.arpa', '128-25.41.122.212.in-addr.arpa',
            '22.172.in-addr.arpa', 'gocept.net-internal'])

    def test_list_external_zones(self):
        self.assertEqual(sorted(z.name for z in self.z.all_external_zones()), [
            '1.0.1.0.8.4.2.0.2.0.a.2.ip6.arpa',
            '125.62.195.in-addr.arpa', '128-25.41.122.212.in-addr.arpa',
            'gocept.net-external'])

    def test_add_reverse_to_correct_zone(self):
        self.z.add_reverse(ip.IPAddress('2a02:248:101:63::5b'), 'vm00.ipv6')
        self.assertEqual('vm00.ipv6.gocept.net.',
                         self.z.reverse_zones[ip.IPNetwork(
                             '2a02:248:101::/48')].records[0].value)

    def test_add_reverse_fails_if_address_does_not_fit_any_zone(self):
        with self.assertRaises(KeyError):
            self.z.add_reverse(ip.IPAddress('192.0.1.2'), 'vm00')

    def test_parse_only_one_nameserver_from_config(self):
        self.c['settings']['nameservers'] = 'ns1.gocept.com'
        zones = Zones(self.c)
        self.assertEqual(['ns1.gocept.com'], zones.nameservers)


class ZonesConfigSaveTest(unittest.TestCase):

    def setUp(self):
        self.c = configobj.ConfigObj(pkg_resources.resource_stream(
            __name__, 'fixtures/configure-zones.cfg'))
        self.z = Zones(self.c)
        self.z.config['internal']['zonelist'] = tempfile.NamedTemporaryFile(
            prefix='internal.', delete=False).name
        self.z.config['external']['zonelist'] = tempfile.NamedTemporaryFile(
            prefix='external.', delete=False).name

    def tearDown(self):
        os.unlink(self.z.config['internal']['zonelist'])
        os.unlink(self.z.config['external']['zonelist'])

    def test_update_config_writes_zonelist(self):
        self.z.update_bind_config()
        with open(self.z.config['internal']['zonelist']) as f:
            self.assertIn("""\
zone "gocept.net" IN {
    type master;
    file "/etc/bind/pri/gocept.net-internal.zone";
};
""", f.read())
        with open(self.z.config['external']['zonelist']) as f:
            self.assertIn("""\
zone "gocept.net" IN {
    type master;
    file "/etc/bind/pri/gocept.net-external.zone";
};
""", f.read())

    def test_update_config_should_return_false_if_unchanged(self):
        self.assertTrue(self.z.update_bind_config())
        self.assertFalse(self.z.update_bind_config())

    def test_includes_are_passed_to_forward_zones(self):
        self.assertEqual(['/etc/bind/pri/gocept.net.zone.static'],
                         self.z.external_forward.include)
        self.assertEqual(['/etc/bind/pri/gocept.net.zone.static',
                          '/etc/bind/pri/gocept.net-internal.zone.static'],
                         self.z.internal_forward.include)


class NodeAddrTest(unittest.TestCase):

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_canonical_records_ipv4(self, zones):
        n = NodeAddr('vm00', 'fe', 'whq', ip.IPAddress('195.62.120.10'))
        n.inject_records(zones)
        zones.add_addr.assert_any_call(
            'vm00.fe.whq', ip.IPAddress('195.62.120.10'))
        zones.add_addr.assert_any_call(
            'vm00.fe.whq.ipv4', ip.IPAddress('195.62.120.10'))

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_short_and_canonical_for_srv_ipv4(self, zones):
        n = NodeAddr('vm00', 'srv', 'whq', ip.IPAddress('212.122.41.136'))
        n.inject_records(zones)
        zones.add_addr.assert_any_call('vm00',
                                       ip.IPAddress('212.122.41.136'),
                                       ['vm00.srv.whq'])
        zones.add_addr.assert_any_call('vm00.ipv4',
                                       ip.IPAddress('212.122.41.136'),
                                       ['vm00.srv.whq.ipv4'])

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_canonical_records_ipv6(self, zones):
        n = NodeAddr('vm00', 'fe', 'whq', ip.IPAddress('2a02:248:101:63::5b'))
        n.inject_records(zones)
        zones.add_addr.assert_any_call(
            'vm00.fe.whq', ip.IPAddress('2a02:248:101:63::5b'))
        zones.add_addr.assert_any_call(
            'vm00.fe.whq.ipv6', ip.IPAddress('2a02:248:101:63::5b'))

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_short_and_canonical_for_srv_ipv6(self, zones):
        n = NodeAddr('vm00', 'srv', 'whq', ip.IPAddress('2a02:248:101:63::5b'))
        n.inject_records(zones)
        zones.add_addr.assert_any_call('vm00',
                                       ip.IPAddress('2a02:248:101:63::5b'),
                                       ['vm00.srv.whq'])
        zones.add_addr.assert_any_call('vm00.ipv6',
                                       ip.IPAddress('2a02:248:101:63::5b'),
                                       ['vm00.srv.whq.ipv6'])

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_default_reverse(self, zones):
        n = NodeAddr('vm00', 'srv', 'whq', ip.IPAddress('212.122.42.136'))
        n.inject_records(zones)
        zones.add_reverse.assert_called_with(ip.IPAddress('212.122.42.136'),
                                             'vm00')

    @mock.patch('gocept.net.configure.zones.Zones')
    def test_custom_reverse(self, zones):
        n = NodeAddr('vm00', 'fe', 'whq', ip.IPAddress('195.62.125.10'),
                     reverse='www.example.com.')
        n.inject_records(zones)
        zones.add_reverse.assert_called_with(ip.IPAddress('195.62.125.10'),
                                             'www.example.com.')
