"""Services for the HEOS integration."""

import logging

from pyheos import CommandAuthenticationError, Heos, HeosError
import voluptuous as vol

from homeassistant.config_entries import ConfigEntryState
from homeassistant.core import HomeAssistant, ServiceCall
from homeassistant.exceptions import HomeAssistantError, ServiceValidationError
from homeassistant.helpers import config_validation as cv, issue_registry as ir

from .const import (
    ATTR_PASSWORD,
    ATTR_USERNAME,
    DOMAIN,
    SERVICE_SIGN_IN,
    SERVICE_SIGN_OUT,
)
from .coordinator import HeosConfigEntry

_LOGGER = logging.getLogger(__name__)

HEOS_SIGN_IN_SCHEMA = vol.Schema(
    {vol.Required(ATTR_USERNAME): cv.string, vol.Required(ATTR_PASSWORD): cv.string}
)

HEOS_SIGN_OUT_SCHEMA = vol.Schema({})


def register(hass: HomeAssistant) -> None:
    """Register HEOS services."""
    hass.services.async_register(
        DOMAIN,
        SERVICE_SIGN_IN,
        _sign_in_handler,
        schema=HEOS_SIGN_IN_SCHEMA,
    )
    hass.services.async_register(
        DOMAIN,
        SERVICE_SIGN_OUT,
        _sign_out_handler,
        schema=HEOS_SIGN_OUT_SCHEMA,
    )


def _get_controller(hass: HomeAssistant) -> Heos:
    """Get the HEOS controller instance."""
    _LOGGER.warning(
        "Actions 'heos.sign_in' and 'heos.sign_out' are deprecated and will be removed in the 2025.8.0 release"
    )
    ir.async_create_issue(
        hass,
        DOMAIN,
        "sign_in_out_deprecated",
        breaks_in_ha_version="2025.8.0",
        is_fixable=False,
        severity=ir.IssueSeverity.WARNING,
        translation_key="sign_in_out_deprecated",
    )

    entry: HeosConfigEntry | None = (
        hass.config_entries.async_entry_for_domain_unique_id(DOMAIN, DOMAIN)
    )

    if not entry or not entry.state == ConfigEntryState.LOADED:
        raise HomeAssistantError(
            translation_domain=DOMAIN, translation_key="integration_not_loaded"
        )
    return entry.runtime_data.heos


async def _sign_in_handler(service: ServiceCall) -> None:
    """Sign in to the HEOS account."""
    controller = _get_controller(service.hass)
    username = service.data[ATTR_USERNAME]
    password = service.data[ATTR_PASSWORD]
    try:
        await controller.sign_in(username, password)
    except CommandAuthenticationError as err:
        raise ServiceValidationError(
            translation_domain=DOMAIN, translation_key="sign_in_auth_error"
        ) from err
    except HeosError as err:
        raise HomeAssistantError(
            translation_domain=DOMAIN,
            translation_key="sign_in_error",
            translation_placeholders={"error": str(err)},
        ) from err


async def _sign_out_handler(service: ServiceCall) -> None:
    """Sign out of the HEOS account."""
    controller = _get_controller(service.hass)
    try:
        await controller.sign_out()
    except HeosError as err:
        raise HomeAssistantError(
            translation_domain=DOMAIN,
            translation_key="sign_out_error",
            translation_placeholders={"error": str(err)},
        ) from err
