import itertools
import json
import logging
import re
from abc import ABC, abstractmethod
from datetime import datetime
from pathlib import Path
from typing import (
    Any,
    Iterable,
    Literal,
    NamedTuple,
    Optional,
    TypeGuard,
    Union,
    get_args,
)


import pandas as pd
import requests
from pydantic import BaseModel, Field, ConfigDict

from epx.core.cloud.auth import platform_api_headers
from epx.core.types.common import RunRequest
from epx.core.utils.config import read_auth_config
from epx.core.utils.http_request import retry_request
from epx.run.fs import FileFinderF11, FileFinderF10, FileFinder
from epx.run.exec.cloud.strategy import (
    UnauthorizedUserError,
    ForbiddenResponse,
)

logger = logging.getLogger(__name__)

LogLevel = Literal["DEBUG", "INFO", "WARNING", "ERROR"]
StatusName = Literal["NOT_STARTED", "QUEUED", "RUNNING", "ERROR", "DONE"]


class _GetRunResponse(BaseModel):
    """Response object from the /runs endpoint for an individual run.

    Attributes
    ----------
    id : int
        Unique ID for the run.
    user_id : int
        The ID of the user who submitted the run.
    created_ts : datetime
        The timestamp when the run was created.
    request : RunRequest
        A copy of the originating request object that the response relates to.
    pod_phase : Literal["Pending", "Running", "Succeeded", "Failed", "Unknown"]
        Indicates the phase of the associated pod.
    containerStatus: Any
        Contains container's info of the run.
    user_deleted : bool
        Indicates if the run has been deleted.
    status : Literal["NOT_STARTED", "QUEUED", "RUNNING", "ERROR", "DONE"]
        Textual description of the status of the run.
    job_id : int
        The ID of the job associated with the run.
    epx_client_version : str
        Represents the client version used to initiate the run.
    """

    model_config = ConfigDict(populate_by_name=True)

    id: int = Field(alias="id")
    user_id: int = Field(alias="userId")
    created_ts: datetime = Field(alias="createdTs")
    request: RunRequest = Field(alias="request")
    pod_phase: Optional[
        Literal["Pending", "Running", "Succeeded", "Failed", "Unknown"]
    ] = Field(default=None, alias="podPhase")
    containerStatus: Optional[Any] = None
    user_deleted: bool = Field(alias="userDeleted")
    status: Literal["NOT_STARTED", "QUEUED", "RUNNING", "ERROR", "DONE"]
    job_id: int = Field(alias="jobId")
    epx_client_version: Optional[str] = Field(default=None, alias="epxClientVersion")


class _GetRunsResponseBody(BaseModel):
    """Response object for a batch of submitted runs from the /runs endpoint.

    Attributes
    ----------
    run_responses : list[RunResponse]
        Collection of responses for individual runs in the originating
        request.
    """

    model_config = ConfigDict(populate_by_name=True)

    runs: list[_GetRunResponse] = Field(alias="runs")


class LogItem(NamedTuple):
    """An individual entry in the logs generated by FRED.

    Attributes
    ----------
    level : LogLevel
        The log level of the message, e.g. `INFO`, `ERROR`, etc.
    time : datetime
        The time that the message was reported at.
    message : str
        The log message.
    """

    level: LogLevel
    time: datetime
    message: str


class FREDRunStatus(ABC):
    """Base class for returning run status information."""

    def __init__(self, file_finder: FileFinder, run_id: int | None):
        self._base_file_finder = file_finder
        self.run_id = run_id

    @property
    @abstractmethod
    def logs(self) -> pd.DataFrame:
        """Return a collection of log entries output by FRED.

        Returns
        -------
        pd.DataFrame
            Collection of individual log entries generated by FRED during the
            run.
        """
        ...

    @property
    def name(self) -> StatusName:
        """Return a string summarizing the job status.

        Returns
        -------
        Status
            A string indicating the status of the job, one of: `"NOT_STARTED"`,
            `"RUNNING"`, `"ERROR"`, `"QUEUED"`or `"DONE"`.

        Raises
        ------
        UnauthorizedUserError
            If the user does not have sufficient privileges to perform the
            specified action on FRED Cloud.
        ConnectionError
            If network connection issues.
        RuntimeError
            If a FRED Cloud server error occurs.
        """

        # Call api here to get StatusName
        endpoint_url = f"{read_auth_config('api-url')}/runs"
        params = {"id": str(self.run_id)}
        headers = platform_api_headers()

        try:
            # Get request for a run to be executed to FRED Cloud API
            response = retry_request(
                method="GET",
                url=endpoint_url,
                headers=headers,
                params=params,
            )
        except requests.exceptions.RequestException as e:
            raise e
        # Check HTTP response status code and raise exceptions as appropriate
        if not response.ok:
            if response.status_code == requests.codes.forbidden:
                raise UnauthorizedUserError(
                    ForbiddenResponse.model_validate_json(response.text).description
                )
            else:
                raise RuntimeError(f"FRED Cloud error code: {response.status_code}")

        response_payload = json.loads(response.text)
        logger.debug(f"Payload: {response.text}")

        if len(response_payload["runs"]) == 0:
            return "NOT_STARTED"
        run = response_payload["runs"][0]
        return run["status"]

    def __str__(self):
        return self.name

    @staticmethod
    def _is_valid_log_level(level: str) -> TypeGuard[LogLevel]:
        """Helper method for validating that a string is a valid log level."""
        return level in get_args(LogLevel)


class RunStatusF10(FREDRunStatus):
    """Accessor for information about the status of a run.

    Notes
    -----
    This implementation is for runs of FRED <= 10.1.0.
    """

    def __init__(self, file_finder: FileFinderF10, run_id: int | None):
        """LegacyRunStatus constructor.

        Parameters
        ----------
        file_finder : LegacyFileFinder
            Accessor for filesystem paths of FRED output files for FRED
            <=10.1.0.
        """

        super().__init__(file_finder, run_id)
        self._file_finder = file_finder
        self.run_id = run_id

    @property
    def logs(self) -> pd.DataFrame:
        try:
            log_iterators = [self._parse_status_file(self._file_finder.status)]
        except FileNotFoundError:
            log_iterators = []
        if self._file_finder.errors is not None:
            log_iterators.append(self._parse_errors_file(self._file_finder.errors))
        return pd.DataFrame.from_records(
            tuple(itertools.chain(*log_iterators)), columns=["level", "time", "message"]
        )

    @staticmethod
    def _parse_status_file(p: Path) -> Iterable[LogItem]:
        """Helper method to parse status.txt"""

        file_write_time = datetime.fromtimestamp(p.stat().st_mtime)
        with open(p, "r") as f:
            for line in f.readlines():
                trimmed_line = line.strip()
                if trimmed_line:
                    yield LogItem("INFO", file_write_time, trimmed_line)

    @staticmethod
    def _parse_errors_file(p: Path) -> Iterable[LogItem]:
        """Helper method to parse errors.txt."""

        file_write_time = datetime.fromtimestamp(p.stat().st_mtime)
        with open(p, "r") as f:
            messages = [x.strip() for x in f.read().split("FRED ERROR: ") if x]
        return tuple(LogItem("ERROR", file_write_time, m) for m in messages)

    def __repr__(self):
        return f"RunStatusF10({self._file_finder})"


class RunStatusF11(FREDRunStatus):
    """Accessor for information about the status of a run.

    Notes
    -----
    This implementation is for runs of FRED >= 11.0.0.
    """

    def __init__(self, file_finder: FileFinderF11, run_id: int | None):
        super().__init__(file_finder, run_id)
        self._file_finder = file_finder
        self.run_id = run_id
        self._log_re = re.compile(
            r"^\[(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}.\d{3}Z)\] ([A-Z]*): (.*)$"
        )

    @property
    def logs(self) -> pd.DataFrame:
        def process_log_line(line: str) -> LogItem:
            m = self._log_re.match(line)
            if m is None:
                raise ValueError(f"Invalid logline: {line}")
            level = m.group(2)
            assert self._is_valid_log_level(level)
            return LogItem(
                level,
                datetime.strptime(m.group(1), "%Y-%m-%dT%H:%M:%S.%fZ"),
                m.group(3),
            )

        try:
            with open(self._file_finder.logs, "r") as f:
                log_records = tuple(process_log_line(line) for line in f.readlines())
        except FileNotFoundError:
            log_records = tuple()
        return pd.DataFrame.from_records(
            log_records, columns=["level", "time", "message"]
        )

    def __repr__(self):
        return f"RunStatusF11({self._file_finder})"


class FREDRunStatusFactory:
    """Factory for selecting correct RunStatus implementation."""

    def __init__(
        self, file_finder: Union[FileFinderF10, FileFinderF11], run_id: int | None
    ):
        self.file_finder = file_finder
        self.run_id = run_id

    def build(self) -> FREDRunStatus:
        """Obtain a RunStatus appropriate for the given output directory.

        Raises
        ------
        TypeError
            If given `file_finder` is not a recognized type.
        """

        if self._is_legacy_file_finder(self.file_finder):
            return RunStatusF10(self.file_finder, self.run_id)
        elif self._is_file_finder(self.file_finder):
            return RunStatusF11(self.file_finder, self.run_id)
        else:
            raise TypeError(f"{self.file_finder} is not a valid FileFinder type")

    @staticmethod
    def _is_legacy_file_finder(file_finder: Any) -> TypeGuard[FileFinderF10]:
        if isinstance(file_finder, FileFinderF10):
            return True
        return False

    @staticmethod
    def _is_file_finder(file_finder: Any) -> TypeGuard[FileFinderF11]:
        if isinstance(file_finder, FileFinderF11):
            return True
        return False
