# -*- coding: utf-8 -*-

from tempfile import mkstemp
import os.path
import re

from flask import Blueprint, abort, send_file
from flask_restful import (Resource, Api, reqparse, fields)
from flask_cors import CORS

from wordweaver import __file__ as ww_file
from wordweaver.log import logger
from wordweaver.resources.verb import verb_fields
from wordweaver.resources.pronoun import pronoun_fields
from wordweaver.resources.affix import affix_fields
from wordweaver.buildtools.file_maker import DocxMaker, LatexMaker
from wordweaver.resources import require_appkey
from wordweaver.fst.english_generator import EnglishGenerator
from wordweaver.exceptions import FomaInputException
from wordweaver.fst.encoder import FstEncoder
from wordweaver.fst.decoder import FstDecoder
from wordweaver.config import ENV_CONFIG

fpb = ENV_CONFIG['FOMA_PYTHON_BINDINGS']
if fpb:
    from wordweaver.fst.utils.foma_access_python import foma_access_python as foma_access
else:
    from wordweaver.fst.utils.foma_access import foma_access


DATA_DIR = os.environ.get('WW_DATA_DIR')

if not DATA_DIR:
    logger.warning(
        'WW_DATA_DIR environment variable is not set, using default sample data instead.')
    DATA_DIR = os.path.join(os.path.dirname(ww_file), 'sample', 'data')

FOMABINS_DIR = os.path.join(DATA_DIR, 'fomabins')


conjugation_fields = {
    'root': fields.Nested(verb_fields),
    'aspect': fields.Nested(affix_fields),
    'agent': fields.Nested(pronoun_fields),
    'patient': fields.Nested(pronoun_fields),
    'tmp_affix': fields.Nested(affix_fields),
    'pas': fields.Nested(affix_fields)
}


class ConjugationList(Resource):
    def __init__(self, fp=foma_access(os.path.join(FOMABINS_DIR,
                                                   ENV_CONFIG["fst_filename"]))):
        self.parser = reqparse.RequestParser()
        self.foma = fp
        self.translator = EnglishGenerator()

        self.parser.add_argument(
            'agent', dest='agent',
            type=str, location='args', action='append',
            required=False, help='An agent tag for the conjugation',
        )

        self.parser.add_argument(
            'patient', dest='patient',
            type=str, location='args', action='append',
            required=False, help='A patient tag for the conjugation',
        )

        self.parser.add_argument(
            'aff-option', dest='aff-option',
            type=str, location='args', action='append',
            required=False, help='An affix option tag for the conjugation',
        )

        self.parser.add_argument(
            'root', dest='root',
            type=str, location='args', action='append',
            required=False, help='A verb root tag for the conjugation',
        )

        self.parser.add_argument(
            'offset', dest='offset',
            type=int, location='args', default=0,
            required=False, help='An offset for conjugations with default of 0 - maximum range between offset and limit is 100',
        )

        self.parser.add_argument(
            'limit', dest='limit',
            type=int, location='args', default=5,
            required=False, help='A limit for conjugations with a default of 5 - maximum range between offset and limit is 100',
        )

        self.parser.add_argument(
            'markers', dest='markers',
            type=bool, location='args', default=False,
            required=False, help='Return the marker generated by the FST instead of the normal response. Meant for debugging.',
        )

        self.parser.add_argument(
            'tags', dest='tags',
            type=bool, location='args', default=False,
            required=False, help='Return the tag given to the FST instead of the normal response. Meant for debugging',
        )

        self.parser.add_argument(
            'plain', dest='plain',
            type=bool, location='args', default=False,
            required=False, help='Return plain text. Meant for debugging',
        )

        self.parser.add_argument(
            'docx', dest='docx',
            type=bool, location='args', default=False,
            required=False, help='Return docx file.',
        )

        self.parser.add_argument(
            'latex', dest='latex',
            type=bool, location='args', default=False,
            required=False, help='Return latex file.',
        )

    def markerToValues(self, marker):
        decoder = FstDecoder(marker)
        return decoder.returnValuesFromMarkers()

    def mergeTagAndValues(self, tag, value):
        value['root']['tag'] = tag['root']
        value['pronoun']['agent'] = tag['agent']
        value['pronoun']['patient'] = tag['patient']
        value['affopt'] = tag['affopt']
        return value

    def returnPlain(self, marker, sep=''):
        vals_pattern = re.compile(r"\^[A-Z][\w\-\']*\^")
        values = re.split(vals_pattern, marker)
        new_value = [x for x in values if x]
        return sep.join(new_value)

    @require_appkey
    def get(self):
        # Get args
        args = self.parser.parse_args()
        conj_range = args['limit'] - args['offset']
        if conj_range > 10:
            abort(403, description="Range between offset and limit {} exceeds maximum allowed. Please contact developers if you require more access.".format(conj_range))

        # Turn args into tags
        tag_maker = FstEncoder(args)
        try:
            tags = tag_maker.return_tags()
        except FomaInputException as e:
            print(("Foma Error" + e))
            abort(
                400, description="The FST failed to conjugate. Exception: {}".format(str(e)))

        # start with an empty response list
        response = []

        # Trigger "down" from fst with tags (return verb).
        for tag in tags:
            # get conjugations from FST
            markers = list(self.foma.down(tag['fst']))
            if not markers:
                markers = ['???']
            # translate from the tags
            translation = self.translator.transduce_tags(tag['fst'])
            for m in markers:
                conjugation = {}
                if m != '???' and '+' not in m:
                    conjugation["translation"] = translation
                    conjugation["values"] = self.mergeTagAndValues(tag['http_args'], self.markerToValues(m))
                if 'tags' in args and args['tags']:
                    conjugation['tag'] = tag
                if 'markers' in args and args['markers']:
                    conjugation['marker'] = m
                if 'plain' in args and args['plain']:
                    conjugation['plain_text'] = self.returnPlain(m)
                response.append(conjugation)

        # If you want to build a docx, build and return
        if "docx" in args and args['docx']:
            dm = DocxMaker(response)
            document = dm.export()
            fd, path = mkstemp()
            try:
                document.save(path)
                return send_file(path,
                                 as_attachment = True,
                                 mimetype = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                                 attachment_filename = 'conjugations.docx')
            finally:
                os.remove(path)

        # If you want to build a latex doc, build and return
        if "latex" in args and args['latex']:
            lm=LatexMaker(response)
            latex=lm.export_to_bytes()
            fd, path=mkstemp()
            try:
                with os.fdopen(fd, 'wb') as tmp:
                    # do stuff with temp file
                    tmp.write(latex)
                    tmp.seek(0)
                    return send_file(path,
                                     as_attachment = True,
                                     mimetype = 'text/plain',
                                     attachment_filename = 'conjugations.tex')
            finally:
                os.remove(path)
        # don't filter if debugging
        if ('tags' in args and args['tags']) or ('markers' in args and args['markers']):
            return response
        # filter out all malformed responses without values
        return [r for r in response if 'values' in r]

# Main API

conjugation_api=Blueprint('resources.conjugation', __name__)
CORS(conjugation_api)
api=Api(conjugation_api)

api.add_resource(
    ConjugationList,
    '/conjugations',
    endpoint = 'conjugations'
)

# Secondary API

conjugation_api_2=Blueprint('resources.conjugation2', __name__)
CORS(conjugation_api_2)
api2=Api(conjugation_api_2)

api2.add_resource(
    ConjugationList,
    '/conjugations',
    endpoint = 'conjugations',
    resource_class_kwargs = {'fp': foma_access(os.path.join(FOMABINS_DIR,
                                                          ENV_CONFIG['test_fst_filename']))}
)
