"""
Pantheon Legends - Scanner to Legend Converter
Interactive tool to help convert existing scanners into Legend engines
"""

import os
from datetime import datetime
from typing import Optional

def setup_scanner_as_legend():
    """
    Interactive tool to convert an existing scanner into a Pantheon Legend
    """
    print("🏛️  Pantheon Legends - Scanner to Legend Setup")
    print("=" * 50)
    print("This tool will help you convert your existing scanner into a Legend engine.\n")
    
    # Gather basic information
    scanner_name = input("📊 What's the name of your scanner? (e.g., 'ResonanceBreakout'): ").strip()
    if not scanner_name:
        scanner_name = "MyScanner"
    
    legend_name = f"{scanner_name}Legend"
    
    print(f"\n✅ Creating legend: {legend_name}")
    
    # Ask about scanner characteristics
    print("\n🔍 Tell us about your scanner:")
    
    signal_types = input("What signals does it detect? (e.g., 'breakout, volume_spike, momentum'): ").strip()
    if not signal_types:
        signal_types = "breakout"
    
    timeframes = input("What timeframes does it work with? (e.g., '1m, 5m, 1h'): ").strip()
    if not timeframes:
        timeframes = "1m"
    
    data_needed = input("What data does it need? (e.g., 'price, volume, moving_averages'): ").strip()
    if not data_needed:
        data_needed = "price, volume"
    
    # Generate the legend template
    template = generate_legend_template(
        legend_name=legend_name,
        scanner_name=scanner_name, 
        signal_types=signal_types,
        timeframes=timeframes,
        data_needed=data_needed
    )
    
    # Save to file
    filename = f"{legend_name.lower()}.py"
    with open(filename, 'w') as f:
        f.write(template)
    
    print(f"\n🎉 Success! Created {filename}")
    print("\n📋 Next Steps:")
    print("1. Open the generated file and replace the TODO comments with your scanner logic")
    print("2. Copy your existing scanner functions into the run_async method")
    print("3. Convert your scanner outputs to the 'facts' dictionary format")
    print("4. Test with: python -c \"from legends import *; import your_legend_file\"")
    print("\n💡 Need help? Check the examples in the pantheon-legends documentation!")

def generate_legend_template(legend_name: str, scanner_name: str, signal_types: str, 
                           timeframes: str, data_needed: str) -> str:
    """Generate a Python legend template based on user inputs"""
    
    signals_list = [s.strip() for s in signal_types.split(',')]
    timeframes_list = [t.strip() for t in timeframes.split(',')]
    data_list = [d.strip() for d in data_needed.split(',')]
    
    template = f'''"""
{legend_name} - Converted from {scanner_name} Scanner
Generated by Pantheon Legends scaffold tool
"""

import asyncio
from datetime import datetime
from typing import Dict, Any, Optional
from legends.contracts import ILegendEngine, LegendRequest, LegendEnvelope, QualityMeta, LegendProgress, ProgressCallback

class {legend_name}:
    """
    Legend engine converted from {scanner_name} scanner
    
    Detects: {signal_types}
    Timeframes: {timeframes} 
    Data Required: {data_needed}
    """
    
    @property
    def name(self) -> str:
        return "{scanner_name}"
    
    async def run_async(
        self,
        request: LegendRequest,
        progress_callback: Optional[ProgressCallback] = None
    ) -> LegendEnvelope:
        """
        Convert your scanner logic here
        """
        
        # Report progress
        if progress_callback:
            await progress_callback(LegendProgress(
                legend=self.name,
                stage="fetch",
                percent=10.0,
                note="Fetching market data"
            ))
        
        # TODO: Replace this with your actual scanner data fetching
        symbol = request.symbol
        timeframe = request.timeframe
        as_of = request.as_of
        
        # Simulate data fetching (replace with your data source)
        market_data = await self._fetch_market_data(symbol, timeframe, as_of)
        
        if progress_callback:
            await progress_callback(LegendProgress(
                legend=self.name,
                stage="analyze",
                percent=50.0,
                note="Running scanner analysis"
            ))
        
        # TODO: Replace this with your actual scanner analysis logic
        analysis_results = await self._run_scanner_analysis(market_data)
        
        if progress_callback:
            await progress_callback(LegendProgress(
                legend=self.name,
                stage="generate_signals",
                percent=80.0,
                note="Generating signals"
            ))
        
        # TODO: Convert your scanner outputs to facts dictionary
        facts = self._convert_to_facts(analysis_results)
        
        # TODO: Calculate quality metrics based on your data
        quality = self._calculate_quality(market_data)
        
        if progress_callback:
            await progress_callback(LegendProgress(
                legend=self.name,
                stage="complete",
                percent=100.0,
                note="Analysis complete"
            ))
        
        return LegendEnvelope(
            legend=self.name,
            at=request.as_of,
            tf=request.timeframe,
            facts=facts,
            quality=quality
        )
    
    async def _fetch_market_data(self, symbol: str, timeframe: str, as_of: datetime) -> Dict[str, Any]:
        """
        TODO: Replace with your data fetching logic
        
        Your scanner probably gets data from:
        - API calls (Binance, Coinbase, etc.)
        - Database queries  
        - Real-time feeds
        - File inputs
        
        Return the data in whatever format your scanner expects
        """
        # Simulate async data fetching
        await asyncio.sleep(0.1)
        
        # TODO: Replace with your actual data fetching
        return {{
            "symbol": symbol,
            "timeframe": timeframe,
            "price": 100.0,  # Replace with real price data
            "volume": 1000000,  # Replace with real volume data
            # Add other data your scanner needs: {data_needed}
        }}
    
    async def _run_scanner_analysis(self, market_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        TODO: Copy your main scanner logic here
        
        This is where your existing scanner code goes:
        - Technical indicator calculations
        - Pattern recognition
        - Signal generation
        - Breakout detection
        - etc.
        """
        # Simulate analysis
        await asyncio.sleep(0.1)
        
        # TODO: Replace with your actual scanner analysis
        # Example structure based on signals: {signal_types}
        return {{
{_generate_example_results(signals_list)}
        }}
    
    def _convert_to_facts(self, analysis_results: Dict[str, Any]) -> Dict[str, Any]:
        """
        TODO: Convert your scanner output format to Pantheon facts format
        
        Facts should be a dictionary with your scanner's key findings:
        - Signal strength/confidence
        - Direction (bullish/bearish)
        - Specific values (price targets, volume ratios, etc.)
        - Any other important results
        """
        
        # TODO: Map your scanner results to standardized facts
        facts = {{
            "signals_detected": [],  # List of signal types found
            "primary_signal": None,  # Main signal if any
            "confidence": 0.0,       # Overall confidence (0-1)
            "direction": "neutral",  # "bullish", "bearish", or "neutral"
            # Add scanner-specific facts here
        }}
        
        # TODO: Fill facts from your analysis_results
        # Example:
        # facts["confidence"] = analysis_results.get("confidence", 0.0)
        # facts["direction"] = analysis_results.get("trend", "neutral")
        
        return facts
    
    def _calculate_quality(self, market_data: Dict[str, Any]) -> QualityMeta:
        """
        TODO: Calculate data quality metrics based on your data source
        """
        
        # TODO: Calculate actual quality metrics from your data
        sample_size = len(market_data.get("candles", [100]))  # Number of data points
        freshness_sec = 60.0  # How recent is the data (seconds)
        data_completeness = 1.0  # Percentage of expected data present (0-1)
        
        return QualityMeta(
            sample_size=float(sample_size),
            freshness_sec=freshness_sec,
            data_completeness=data_completeness
        )

# Example usage and testing
async def test_{legend_name.lower()}():
    """Test the converted legend"""
    legend = {legend_name}()
    
    request = LegendRequest(
        symbol="BTC-USD",  # Example symbol
        timeframe="1m",    # Example timeframe  
        as_of=datetime.now()
    )
    
    async def progress_handler(progress: LegendProgress):
        print(f"[{{progress.legend}}] {{progress.stage}}: {{progress.percent:.1f}}% - {{progress.note}}")
    
    result = await legend.run_async(request, progress_handler)
    
    print(f"\\n✅ Legend Analysis Complete!")
    print(f"Legend: {{result.legend}}")
    print(f"Facts: {{result.facts}}")
    print(f"Quality: Sample={{result.quality.sample_size}}, Fresh={{result.quality.freshness_sec}}s")

if __name__ == "__main__":
    print("Testing {legend_name}...")
    asyncio.run(test_{legend_name.lower()}())
'''
    
    return template

def _generate_example_results(signals_list):
    """Generate example analysis results based on signal types"""
    examples = []
    for signal in signals_list:
        signal = signal.strip().lower()
        if 'breakout' in signal:
            examples.append('            "breakout_detected": False,  # TODO: Your breakout detection logic')
            examples.append('            "breakout_strength": 0.0,    # TODO: Strength of breakout (0-1)')
        elif 'volume' in signal:
            examples.append('            "volume_spike": False,       # TODO: Your volume spike detection')
            examples.append('            "volume_ratio": 1.0,        # TODO: Current vs average volume')
        elif 'momentum' in signal:
            examples.append('            "momentum_signal": "neutral", # TODO: Your momentum analysis')
            examples.append('            "momentum_strength": 0.0,    # TODO: Momentum strength')
        else:
            examples.append(f'            "{signal}_signal": None,     # TODO: Your {signal} analysis')
    
    if not examples:
        examples.append('            "signal_detected": False,      # TODO: Your main signal detection')
    
    return '\n'.join(examples)
