"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const copy_1 = require("../../lib/fs/copy");
const follow_mode_1 = require("../../lib/fs/follow-mode");
function tree(dir, depth = '') {
    const lines = new Array();
    for (const file of fs.readdirSync(dir).sort()) {
        const filePath = path.join(dir, file);
        const stat = fs.lstatSync(filePath);
        if (stat.isSymbolicLink()) {
            const linkDest = fs.readlinkSync(filePath);
            lines.push(depth + file + ' => ' + linkDest);
        }
        else if (stat.isDirectory()) {
            lines.push(depth + file + ' (D)');
            lines.push(...tree(filePath, depth + '    '));
        }
        else {
            lines.push(depth + file);
        }
    }
    return lines;
}
module.exports = {
    'Default: copies all files and subdirectories, with default follow mode is "External"'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir);
        // THEN
        test.deepEqual(tree(outdir), [
            'external-link.txt',
            'file1.txt',
            'local-link.txt => file1.txt',
            'subdir (D)',
            '    file2.txt',
            'subdir2 (D)',
            '    empty-subdir (D)',
            '        .hidden',
            '    subdir3 (D)',
            '        file3.txt'
        ]);
        test.done();
    },
    'Always: follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.ALWAYS
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt',
            'local-dir-link (D)',
            '    file-in-subdir.txt',
            'local-link.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'Never: do not follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.NEVER
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link => ../test1/subdir',
            'external-link.txt => ../test1/subdir2/subdir3/file3.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'External: follow only external symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.EXTERNAL
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'exclude'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir, {
            exclude: [
                '*',
                '!subdir2',
                '!subdir2/**/*',
                '.*'
            ]
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'subdir2 (D)',
            '    empty-subdir (D)',
            '    subdir3 (D)',
            '        file3.txt'
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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