(ns sci.impl.parser
  {:no-doc true}
  (:refer-clojure :exclude [read-string eval])
  (:require
   [clojure.string :as str]
   [clojure.tools.reader.reader-types :as r]
   [edamame.core :as edamame]
   [sci.impl.interop :as interop]
   [sci.impl.types :as types]
   [sci.impl.utils :as utils]))

#?(:clj (set! *warn-on-reflection* true))

(def ^:const eof :sci.impl.parser.edamame/eof)

(def read-eval
  (utils/new-var '*read-eval* false {:ns utils/clojure-core-ns
                                    :dynamic true}))

(def data-readers
  (utils/new-var '*data-readers* {}
                {:ns utils/clojure-core-ns
                 :dynamic true}))

(def default-data-reader-fn
  (utils/new-var '*default-data-reader-fn* nil
                {:ns utils/clojure-core-ns
                 :dynamic true}))

(def reader-resolver
  (utils/new-var '*reader-resolver* nil
                {:ns utils/clojure-core-ns
                 :dynamic true}))

(def default-opts
  (edamame/normalize-opts
   {:all true
    :row-key :line
    :col-key :column
    :read-cond :allow
    :location? seq?
    :end-location false}))

(defn var->sym [v]
  (when-let [m (meta v)]
    (if (:sci/record m)
      (-> (deref v)
          str
          symbol)
      (when-let [var-name (:name m)]
        (when-let [ns (:ns m)]
          (symbol (str (types/getName ns))
                  (str var-name)))))))

(defn fully-qualify [ctx sym]
  (let [env @(:env ctx)
        sym-ns (when-let [n (namespace sym)]
                 (symbol n))
        sym-name-str (name sym)
        current-ns (utils/current-ns-name)
        current-ns-str (str current-ns)
        namespaces (get env :namespaces)
        the-current-ns (get namespaces current-ns)
        aliases (:aliases the-current-ns)
        ret (if-not sym-ns
              (or (when-let [refers (:refers the-current-ns)]
                    (when-let [v (get refers sym)]
                      (var->sym v)))
                  (when-let [v (get the-current-ns sym)]
                    (var->sym v))
                  (when (or (and (contains? (get namespaces 'clojure.core) sym)
                                 ;; only valid when the symbol isn't excluded
                                 (not (some-> the-current-ns
                                              :refer
                                              (get 'clojure.core)
                                              :exclude
                                              (contains? sym ))))
                            (contains? utils/ana-macros sym))
                    (symbol "clojure.core" sym-name-str))
                  (interop/fully-qualify-class ctx sym)
                  ;; all unresolvable symbols all resolved in the current namespace
                  (if (str/includes? sym-name-str ".")
                    sym ;; unresolved class
                    (symbol current-ns-str sym-name-str)))
              (if (get-in env [:namespaces sym-ns])
                sym
                (if-let [ns (get aliases sym-ns)]
                  (symbol (str ns) sym-name-str)
                  sym)))]
    ret))

(defn throw-eval-read [_]
  (throw (ex-info "EvalReader not allowed when *read-eval* is false."
                  {:type :sci.error/parse})))

(defn auto-resolve [ctx opts]
  (or (:auto-resolve opts)
      (let [env (:env ctx)
            env-val @env
            current-ns (utils/current-ns-name)
            the-current-ns (get-in env-val [:namespaces current-ns])
            aliases (:aliases the-current-ns)
            auto-resolve (assoc aliases :current current-ns)]
        auto-resolve)))

(defn get-line-number [reader]
  (r/get-line-number reader))

(defn get-column-number [reader]
  (r/get-column-number reader))

(defn parse-next
  ([ctx r]
   (parse-next ctx r nil))
  ([ctx r opts]
   (let [features (:features ctx)
         readers (:readers ctx)
         readers (if (utils/var? readers) @readers readers)
         auto-resolve (auto-resolve ctx opts)
         parse-opts (cond-> (assoc default-opts
                                   :features features
                                   :auto-resolve auto-resolve
                                   :syntax-quote {:resolve-symbol #(fully-qualify ctx %)}
                                   :readers (fn [t]
                                              (or (and readers (readers t))
                                                  (@data-readers t)
                                                  (some-> (@utils/eval-resolve-state ctx {} t)
                                                          meta
                                                          :sci.impl.record/map-constructor)
                                                  (when-let [f @default-data-reader-fn]
                                                    (fn [form]
                                                      (f t form)))))
                                   :read-eval (if @read-eval
                                                (fn [x]
                                                  (utils/eval ctx x))
                                                throw-eval-read))
                      opts (merge opts))
         ret (try (let [v (edamame/parse-next r parse-opts)]
                    (if (utils/kw-identical? v :edamame.core/eof)
                      eof
                      (if (symbol? v)
                        (vary-meta v assoc
                                   :line (get-line-number r)
                                   :column (- (get-column-number r)
                                              #?(:clj (.length (str v))
                                                 :cljs (.-length (str v)))))
                        v)))
                  (catch #?(:clj clojure.lang.ExceptionInfo
                            :cljs cljs.core/ExceptionInfo) e
                    (throw (ex-info #?(:clj (.getMessage e)
                                       :cljs (.-message e))
                                    (assoc (ex-data e)
                                           :type :sci.error/parse
                                           :phase "parse"
                                           :file @utils/current-file)
                                    e))))]
     ret)))

(defn reader [x]
  #?(:clj (r/indexing-push-back-reader (r/push-back-reader x))
     :cljs (let [string-reader (r/string-reader x)
                 buf-len 1
                 pushback-reader (r/PushbackReader. string-reader
                                                    (object-array buf-len)
                                                    buf-len buf-len)]
             (r/indexing-push-back-reader pushback-reader))))

(defn parse-string
  ([ctx s]
   (let [r (reader s)
         v (parse-next ctx r)]
     (if (utils/kw-identical? eof v) nil v))))

;;;; Scratch

(comment
  )
