"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagManager = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_resource_1 = require("./cfn-resource");
const lazy_1 = require("./lazy");
/**
 * Standard tags are a list of { key, value } objects
 */
class StandardFormatter {
    parseTags(cfnPropertyTags, priority) {
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        const tags = [];
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined || tag.value === undefined) {
                throw new Error(`Invalid tag input expected {key, value} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * ASG tags are a list of { key, value, propagateAtLaunch } objects
 */
class AsgFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value, propagateAtLaunch} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined ||
                tag.value === undefined ||
                tag.propagateAtLaunch === undefined) {
                throw new Error(`Invalid tag input expected {key, value, propagateAtLaunch} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
                applyToLaunchedInstances: !!tag.propagateAtLaunch,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
                propagateAtLaunch: tag.applyToLaunchedInstances !== false,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * Some CloudFormation constructs use a { key: value } map for tags
 */
class MapFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (Array.isArray(cfnPropertyTags) || typeof (cfnPropertyTags) !== 'object') {
            throw new Error(`Invalid tag input expected map of {key: value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const [key, value] of Object.entries(cfnPropertyTags)) {
            tags.push({
                key,
                value: `${value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = {};
        for (const tag of tags) {
            cfnTags[`${tag.key}`] = `${tag.value}`;
        }
        return Object.keys(cfnTags).length === 0 ? undefined : cfnTags;
    }
}
/**
 * StackTags are of the format { Key: key, Value: value }
 */
class KeyValueFormatter {
    parseTags(keyValueTags, priority) {
        const tags = [];
        for (const key in keyValueTags) {
            if (keyValueTags.hasOwnProperty(key)) {
                const value = keyValueTags[key];
                tags.push({
                    key,
                    value,
                    priority,
                });
            }
        }
        return tags;
    }
    formatTags(unformattedTags) {
        const tags = [];
        unformattedTags.forEach(tag => {
            tags.push({
                Key: tag.key,
                Value: tag.value,
            });
        });
        return tags.length > 0 ? tags : undefined;
    }
}
class NoFormat {
    parseTags(_cfnPropertyTags) {
        return [];
    }
    formatTags(_tags) {
        return undefined;
    }
}
let _tagFormattersCache;
/**
 * Access tag formatters table
 *
 * In a function because we're in a load cycle with cfn-resource that defines `TagType`.
 */
function TAG_FORMATTERS() {
    return _tagFormattersCache !== null && _tagFormattersCache !== void 0 ? _tagFormattersCache : (_tagFormattersCache = {
        [cfn_resource_1.TagType.AUTOSCALING_GROUP]: new AsgFormatter(),
        [cfn_resource_1.TagType.STANDARD]: new StandardFormatter(),
        [cfn_resource_1.TagType.MAP]: new MapFormatter(),
        [cfn_resource_1.TagType.KEY_VALUE]: new KeyValueFormatter(),
        [cfn_resource_1.TagType.NOT_TAGGABLE]: new NoFormat(),
    });
}
;
/**
 * TagManager facilitates a common implementation of tagging for Constructs
 *
 * Normally, you do not need to use this class, as the CloudFormation specification
 * will indicate which resources are taggable. However, sometimes you will need this
 * to make custom resources taggable. Used `tagManager.renderedTags` to obtain a
 * value that will resolve to the tags at synthesis time.
 *
 * @example
 * import * as cdk from '@aws-cdk/core';
 *
 * class MyConstruct extends cdk.Resource implements cdk.ITaggable {
 *   public readonly tags = new cdk.TagManager(cdk.TagType.KEY_VALUE, 'Whatever::The::Type');
 *
 *   constructor(scope: cdk.Construct, id: string) {
 *     super(scope, id);
 *
 *     new cdk.CfnResource(this, 'Resource', {
 *       type: 'Whatever::The::Type',
 *       properties: {
 *         // ...
 *         Tags: this.tags.renderedTags,
 *       },
 *     });
 *   }
 * }
 *
 */
class TagManager {
    constructor(tagType, resourceTypeName, tagStructure, options = {}) {
        this.tags = new Map();
        this.priorities = new Map();
        this.initialTagPriority = 50;
        jsiiDeprecationWarnings._aws_cdk_core_TagType(tagType);
        jsiiDeprecationWarnings._aws_cdk_core_TagManagerOptions(options);
        this.resourceTypeName = resourceTypeName;
        this.tagFormatter = TAG_FORMATTERS()[tagType];
        if (tagStructure !== undefined) {
            this._setTag(...this.tagFormatter.parseTags(tagStructure, this.initialTagPriority));
        }
        this.tagPropertyName = options.tagPropertyName || 'tags';
        this.renderedTags = lazy_1.Lazy.any({ produce: () => this.renderTags() });
    }
    /**
     * Check whether the given construct is Taggable
     */
    static isTaggable(construct) {
        return construct.tags !== undefined;
    }
    /**
     * Adds the specified tag to the array of tags
     *
     */
    setTag(key, value, priority = 0, applyToLaunchedInstances = true) {
        // This method mostly exists because we don't want to expose the 'Tag' type used (it will be confusing
        // to users).
        this._setTag({ key, value, priority, applyToLaunchedInstances });
    }
    /**
     * Removes the specified tag from the array if it exists
     *
     * @param key The tag to remove
     * @param priority The priority of the remove operation
     */
    removeTag(key, priority) {
        if (priority >= (this.priorities.get(key) || 0)) {
            this.tags.delete(key);
            this.priorities.set(key, priority);
        }
    }
    /**
     * Renders tags into the proper format based on TagType
     *
     * This method will eagerly render the tags currently applied. In
     * most cases, you should be using `tagManager.renderedTags` instead,
     * which will return a `Lazy` value that will resolve to the correct
     * tags at synthesis time.
     */
    renderTags() {
        return this.tagFormatter.formatTags(this.sortedTags);
    }
    /**
     * Render the tags in a readable format
     */
    tagValues() {
        const ret = {};
        for (const tag of this.sortedTags) {
            ret[tag.key] = tag.value;
        }
        return ret;
    }
    /**
     * Determine if the aspect applies here
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     */
    applyTagAspectHere(include, exclude) {
        if (exclude && exclude.length > 0 && exclude.indexOf(this.resourceTypeName) !== -1) {
            return false;
        }
        if (include && include.length > 0 && include.indexOf(this.resourceTypeName) === -1) {
            return false;
        }
        return true;
    }
    /**
     * Returns true if there are any tags defined
     */
    hasTags() {
        return this.tags.size > 0;
    }
    _setTag(...tags) {
        for (const tag of tags) {
            if (tag.priority >= (this.priorities.get(tag.key) || 0)) {
                this.tags.set(tag.key, tag);
                this.priorities.set(tag.key, tag.priority);
            }
        }
    }
    get sortedTags() {
        return Array.from(this.tags.values()).sort((a, b) => a.key.localeCompare(b.key));
    }
}
exports.TagManager = TagManager;
_a = JSII_RTTI_SYMBOL_1;
TagManager[_a] = { fqn: "@aws-cdk/core.TagManager", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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