# Метрики ошибок для оценки качества моделей

Этот модуль содержит набор функций для вычисления различных метрик ошибок, часто используемых при оценке качества предсказаний в задачах регрессии и прогнозирования. Каждая функция принимает два одномерных итерируемых объекта (например, списки или массивы NumPy) – реальные значения (`y_true`) и предсказанные значения (`y_pred`).

> **Важно:** Все функции требуют, чтобы количество элементов в `y_true` и `y_pred` было одинаковым и больше одного.

---

## Список функций

### 1. Средняя абсолютная ошибка (MAE)

**Функция:** `mean_absolute_error(y_true, y_pred)`

**Описание:**
Вычисляет среднюю абсолютную ошибку – среднее значение абсолютных разностей между предсказанными и фактическими значениями.

**Формула:**

$$
\text{MAE} = \frac{1}{n} \sum_{i=1}^{n} \left| y_i - \hat{y}_i \right|
$$

**Частые случаи использования:**

- Оценка точности регрессионных моделей.
- Применяется, когда важна интерпретируемость ошибки в тех же единицах, что и целевая переменная.

**Пример использования:**

```python
y_true = [3, -0.5, 2, 7]
y_pred = [2.5, 0.0, 2, 8]
mae = mean_absolute_error(y_true, y_pred)
print("MAE:", mae)
```

---




### 2. Среднеквадратичная ошибка (MSE)

**Функция:** `mean_squared_error(y_true, y_pred)`

**Описание:**
Вычисляет среднеквадратичную ошибку – среднее значение квадратов разностей между предсказанными и фактическими значениями. MSE более чувствительна к выбросам за счёт возведения ошибок в квадрат.

**Формула:**

$$
\text{MSE} = \frac{1}{n} \sum_{i=1}^{n} \left( y_i - \hat{y}_i \right)^2
$$

**Частые случаи использования:**

- Оптимизация моделей регрессии (например, в линейной регрессии).
- Ситуации, где крупные ошибки должны иметь больший вклад в итоговую ошибку.

**Пример использования:**

```python
mse = mean_squared_error(y_true, y_pred)
print("MSE:", mse)
```


### 3. Средняя абсолютная процентная ошибка (MAPE)

**Функция:** `mean_absolute_percentage_error(y_true, y_pred)`

**Описание:**
MAPE вычисляет относительную ошибку, показывая среднее процентное отклонение предсказанных значений от реальных.

**Формула:**

$$
\text{MAPE} = \frac{1}{n} \sum_{i=1}^{n} \left| \frac{y_i - \hat{y}_i}{y_i} \right|
$$

**Замечание:**
Функция не умножает результат на 100, поэтому MAPE возвращается в виде десятичной дроби (например, 0.15 соответствует 15%). Также следует учитывать, что при $ y_i = 0 $ возникает деление на ноль.

**Частые случаи использования:**

- Прогнозирование временных рядов.
- Оценка моделей, где важны относительные, а не абсолютные отклонения.

**Пример использования:**

```python
mape = mean_absolute_percentage_error(y_true, y_pred)
print("MAPE:", mape)
```

---

### 4. Симметричная средняя абсолютная процентная ошибка (SMAPE)

**Функция:** `symmetric_mean_absolute_percentage_error(y_true, y_pred)`

**Описание:**
SMAPE корректирует классическую MAPE, симметризируя вклад ошибок, что позволяет избежать слишком больших значений при малых суммах $ y_i + \hat{y}_i $.

**Формула:**

$$
\text{SMAPE} = \frac{1}{n} \sum_{i=1}^{n} \frac{2 \left| y_i - \hat{y}_i \right|}{y_i + \hat{y}_i}
$$

**Частые случаи использования:**

- Прогнозирование, когда важно учитывать как завышенные, так и заниженные ошибки.
- Сценарии, где значения целевой переменной могут быть малы, а стандартная MAPE может давать неоправданно высокие значения.

**Пример использования:**

```python
smape = symmetric_mean_absolute_percentage_error(y_true, y_pred)
print("SMAPE:", smape)
```

---

### 5. Взвешенная средняя процентная ошибка (WAPE)

**Функция:** `weighted_average_percentage_error(y_true, y_pred)`

**Описание:**
WAPE вычисляется как отношение суммы абсолютных ошибок к сумме абсолютных значений реальных данных, что позволяет оценить ошибку с учётом величины самих данных.

**Формула:**

$$
\text{WAPE} = \frac{\sum_{i=1}^{n} \left| y_i - \hat{y}_i \right|}{\sum_{i=1}^{n} \left| y_i \right|}

$$

**Частые случаи использования:**

- Прогнозирование спроса и продаж, где важно учитывать общий объём.
- Сценарии с неравномерными величинами целевой переменной.

**Пример использования:**

```python
wape = weighted_average_percentage_error(y_true, y_pred)
print("WAPE:", wape)
```

---

### 6. Среднеквадратичная логарифмическая ошибка (MSLE)

**Функция:** `mean_squared_logarithmic_error(y_true, y_pred, c=1)`

**Описание:**
MSLE измеряет разницу между логарифмами фактических и предсказанных значений. Смещение $ c $ добавляется для избежания логарифма нуля или отрицательных значений.

**Формула (предполагаемая):**

$$
\text{MSLE} = \frac{1}{n} \sum_{i=1}^{n} \left( \log(y_i + c) - \log(\hat{y}_i + c) \right)^2
$$

> **Замечание:** Обратите внимание, что в представленном коде реализация может не учитывать возведение в квадрат, поэтому рекомендуется проверить корректность вычислений.

**Частые случаи использования:**

- Модели, где важна относительная разница между значениями, особенно при экспоненциальном росте.
- Прогнозирование экономических или финансовых показателей, где абсолютные различия менее информативны.

**Пример использования:**

```python
msle = mean_squared_logarithmic_error(y_true, y_pred, c=1)
print("MSLE:", msle)
```

---

### 7. Корень среднеквадратичной логарифмической ошибки (RMSLE)

**Функция:** `root_mean_squared_logarithmic_error(y_true, y_pred, c=1)`

**Описание:**
RMSLE — это квадратный корень из MSLE, что позволяет вернуть ошибку в том же масштабе, что и логарифмы.

**Формула:**

$$
\text{RMSLE} = \sqrt{\frac{1}{n} \sum_{i=1}^{n} \left( \log(y_i + c) - \log(\hat{y}_i + c) \right)^2}
$$

**Частые случаи использования:**

- Аналогично MSLE, применяется в случаях экспоненциального роста.
- Когда необходимо уменьшить влияние крупных выбросов.

**Пример использования:**

```python
rmsle = root_mean_squared_logarithmic_error(y_true, y_pred, c=1)
print("RMSLE:", rmsle)
```

---

### 8. Медианная абсолютная ошибка (MedAE)

**Функция:** `median_absolute_error(y_true, y_pred, multioutput="uniform_average")`

**Описание:**
Вычисляет медиану абсолютных ошибок между предсказанными и реальными значениями. Опция `multioutput` позволяет выбрать способ агрегации ошибок (либо вернуть сырые значения для каждого выхода, либо вычислить их равномерное среднее).

**Формула:**

$$
\text{MedAE} = \text{median}\left(\left| y_i - \hat{y}_i \right|\right)
$$

**Частые случаи использования:**

- Сценарии, где требуется устойчивость к выбросам, поскольку медиана менее чувствительна к аномальным значениям по сравнению со средним.
- Оценка моделей, работающих с мультивыходными данными.

**Пример использования:**

```python
medae = median_absolute_error(y_true, y_pred, multioutput="uniform_average")
print("MedAE:", medae)
```

---

### 9. Максимальная ошибка (ME)

**Функция:** `max_error(y_true, y_pred)`

**Описание:**
Вычисляет максимальное отклонение между предсказанными и фактическими значениями – то есть, наибольшую абсолютную ошибку.

**Формула:**

$$
\text{ME} = \max_{1 \leq i \leq n} \left| y_i - \hat{y}_i \right|
$$

**Частые случаи использования:**

- Оценка худшего случая ошибки.
- Анализ устойчивости модели и выявление критических ошибок.

**Пример использования:**

```python
max_err = max_error(y_true, y_pred)
print("Max Error:", max_err)
```

---

## Общие рекомендации по использованию

- **Валидация входных данных:** Все функции проверяют, что `y_true` и `y_pred` являются одномерными итерируемыми объектами с более чем одним элементом. Перед использованием функций убедитесь, что данные подготовлены корректно.
- **Выбор метрики:**

  - **MAE** и **MSE** часто используются для базовой оценки моделей регрессии.
  - **MAPE** и **SMAPE** полезны, когда важна относительная ошибка, однако следует быть осторожными с нулевыми значениями в `y_true`.
  - **WAPE** помогает учитывать масштаб данных при сравнении моделей.
  - **MSLE** и **RMSLE** предпочтительны при экспоненциальном росте или когда важно учитывать относительную разницу.
  - **MedAE** и **ME** позволяют оценить устойчивость модели и выявить крайние случаи ошибок.
- **Обработка ошибок:** Некоторые функции (например, MSLE и RMSLE) генерируют исключение `ValueError`, если после добавления смещения $ c $ значения становятся неположительными. Это помогает избежать ошибок при вычислении логарифмов.

---

## Зависимости

- Python 3.x
- [NumPy](https://numpy.org/) – библиотека для работы с массивами и выполнения векторизированных операций.

---

## Заключение

Данный модуль предоставляет удобный набор функций для вычисления основных метрик ошибок, что делает его полезным инструментом при анализе и сравнении моделей прогнозирования и регрессии. Выбирайте наиболее подходящую метрику в зависимости от специфики задачи и характеристик данных.
