# Логистическая регрессия

Этот проект реализует модель логистической регрессии с использованием градиентного спуска для решения задач бинарной классификации. Код написан на Python и включает методы для обучения модели, предсказания классов, вычисления вероятностей принадлежности к классу, а также визуализации разделяющей границы.

---

## Описание

Логистическая регрессия — это алгоритм машинного обучения, используемый для решения задач бинарной классификации. Модель предсказывает вероятность принадлежности объекта к одному из двух классов (0 или 1) на основе входных признаков. В данном случае обучение модели осуществляется с помощью градиентного спуска.

---

## Использование

Для использования класса `LogisticRegression` выполните следующие шаги:

1. Импортируйте необходимые библиотеки:

   ```python
   import numpy as np
   import matplotlib.pyplot as plt
   from scipy.special import expit as sigm
   ```
2. Создайте экземпляр класса `LogisticRegression`.
3. Обучите модель, передав матрицу признаков `X` и вектор меток `y` в метод `fit`.
4. Используйте методы `predict`, `predict_proba` и `accuracy_score` для оценки качества модели.
5. Для двумерных данных можно визуализировать разделяющую границу с помощью метода `plot`.

---

## Методы класса

### 1. `__init__()`

Инициализирует модель. Внутренний параметр `w` (вектор весов) устанавливается в `None`.

---

### 2. `predict_proba(X)`

Вычисляет вероятность принадлежности каждого объекта к классу 1.

- **Аргументы:**
  - `X`: Матрица признаков размера `(n_samples, n_features)`.
- **Возвращает:**
  - `numpy.array`: Вероятности для каждого объекта, shape `(n_samples,)`.

---

### 3. `predict(X, threshold=0.5)`

Предсказывает классы (0 или 1) для объектов на основе заданного порога.

- **Аргументы:**
  - `X`: Матрица признаков размера `(n_samples, n_features)`.
  - `threshold`: Порог классификации (по умолчанию 0.5).
- **Возвращает:**
  - `numpy.array`: Вектор предсказанных классов (0 или 1).

---

### 4. `loss(X, y)`

Вычисляет логарифмическую функцию потерь (log loss).

- **Аргументы:**
  - `X`: Матрица признаков размера `(n_samples, n_features)`.
  - `y`: Вектор истинных меток (0 или 1), shape `(n_samples,)`.
- **Возвращает:**
  - `float`: Среднее значение логарифмической потери.

---

### 5. `fit(X, y, learning_rate=0.1, n_iter=1000)`

Обучает модель с помощью градиентного спуска.

- **Аргументы:**
  - `X`: Матрица признаков размера `(n_samples, n_features)`.
  - `y`: Вектор истинных меток (0 или 1), shape `(n_samples,)`.
  - `learning_rate`: Скорость обучения (по умолчанию 0.1).
  - `n_iter`: Количество итераций обучения (по умолчанию 1000).

---

### 6. `accuracy_score(X, y)`

Вычисляет точность модели (accuracy).

- **Аргументы:**
  - `X`: Матрица признаков.
  - `y`: Вектор истинных меток (0 или 1).
- **Возвращает:**
  - `float`: Доля правильно предсказанных объектов.

---

### 7. `plot(X, y)`

Визуализирует разделяющую границу для двумерных данных.

- **Аргументы:**
  - `X`: Матрица признаков размера `(n_samples, 2)`.
  - `y`: Вектор истинных меток (0 или 1).
- **Примечание:** Метод поддерживает только двумерные данные.

---

## Пример использования

```python
# Генерируем простые данные для двух классов
np.random.seed(0)
n_samples = 100
X_class0 = np.random.randn(n_samples, 2) - 1
X_class1 = np.random.randn(n_samples, 2) + 1
X = np.vstack((X_class0, X_class1))
y = np.array([0]*n_samples + [1]*n_samples)
  
# Создаем и обучаем модель
model = LogisticRegression()
model.fit(X, y, learning_rate=0.1, n_iter=1000)
  
# Выводим потери и точность
print("Final loss:", model.loss(X, y))
print("Accuracy:", model.accuracy_score(X, y))
  
# Визуализируем результат для двумерных данных
model.plot(X, y)
```

---
