# Модуль `base` в пакете `nm`

Этот модуль содержит набор функций для демонстрации и анализа различных аспектов численных вычислений, таких как ошибки округления, потеря точности, переполнение и способы их смягчения.

## Функции

### `perepolnenie()`

**Описание:**

Демонстрирует поведение типов данных при переполнении в Python и NumPy. Переполнение происходит, когда результат арифметической операции превышает максимальное значение, которое может быть представлено данным типом данных.

**Пример использования:**

```python
from matplobblib.nm.base import perepolnenie

perepolnenie()
```

**Вывод:**

```
переполнение
2147483647
-2147483648
1.7976931348623157e+308
0.0
inf
```

**Теоретическая справка:**

1. **Целочисленное переполнение (int32):**

   * Максимальное значение для 32-битного целого: 2`<sup>`31`</sup>` - 1 = 2147483647
   * При добавлении 1 к максимальному значению происходит "wrap-around" (циклический перенос), и результат становится минимальным значением: -2147483648.
   * Это связано с тем, что старший бит используется для представления знака числа.
2. **Переполнение чисел с плавающей точкой (float64):**

   * Максимальное значение: приблизительно 1.8 × 10`<sup>`308`</sup>`.
   * При добавлении небольшого числа к максимальному значению может произойти потеря точности, и результат останется прежним из-за ограниченного количества значащих цифр.
   * При добавлении числа, превышающего максимальное значение (например, удвоение максимального значения), результат становится `inf` (положительная бесконечность).
3. **Отличия от стандартных целых чисел Python:**

   * В отличие от NumPy, стандартные целые числа Python (`int`) не имеют фиксированного размера и могут представлять сколь угодно большие значения без переполнения.
   * NumPy использует типы данных фиксированного размера (например, `int32`, `float64`), что может привести к переполнению, но обеспечивает более эффективное использование памяти и скорость вычислений.

### `poterya_tochnosti()`

**Описание:**

Демонстрирует потерю точности при арифметических операциях с числами с плавающей точкой. Потеря точности возникает при вычитании близких по значению чисел, когда значащие разряды сокращаются, и результат может быть неточным.

**Пример использования:**

```python
from matplobblib.nm.base import poterya_tochnosti

poterya_tochnosti()
```

**Вывод:**

```
потеря точности
1e-09
```

(В зависимости от реализации `float` в интерпретаторе результат может быть `0.0` или `1e-09`.)

**Теоретическая справка:**

1. **Ограниченная точность представления:**

   * Числа с плавающей точкой (обычно `float64` в Python) представляются в формате IEEE 754, который использует ограниченное количество бит для хранения числа (64 бита для `float64`).
   * Это означает, что не все действительные числа могут быть представлены точно; многие числа округляются до ближайшего представимого значения.
   * `float64` обеспечивает примерно 15-17 десятичных знаков точности.
2. **Катастрофическое сокращение (Catastrophic Cancellation):**

   * При вычитании двух близких чисел их наиболее значимые разряды сокращаются, и результат содержит только наименее значимые разряды исходных чисел.
   * Если исходные числа были округлены, ошибка округления может стать доминирующей в результате, что приводит к потере точности.
3. **Пример:**

   * Числа 1.000000001 и 1.000000000 различаются на 1 × 10`<sup>`-9`</sup>`.
   * При представлении в формате `float64` эти числа могут храниться с одинаковым приближением, и их разность может быть вычислена как 0 или значение, близкое к 1 × 10`<sup>`-9`</sup>`, но с возможной потерей точности.

### `oshibka_okrugleniya()`

**Описание:**

Демонстрирует ошибки округления, возникающие при арифметических операциях с числами с плавающей точкой. Ошибки округления возникают из-за того, что десятичные дроби не всегда могут быть точно представлены в двоичной системе счисления, используемой для хранения чисел с плавающей точкой.

**Пример использования:**

```python
from matplobblib.nm.base import oshibka_okrugleniya

oshibka_okrugleniya()
```

**Вывод:**

```
ошибки округления
False
```

(Результат может варьироваться в зависимости от реализации `float` в интерпретаторе.)

**Теоретическая справка:**

1. **Ограничения стандарта IEEE 754:**

   * Стандарт IEEE 754 определяет формат представления чисел с плавающей точкой, который используется в большинстве современных компьютеров.
   * В этом формате числа представляются в виде суммы степеней двойки.
   * Десятичные дроби, такие как 0.1 и 0.2, не могут быть точно представлены в двоичной системе, так как они требуют бесконечного количества двоичных разрядов.
2. **Накопление ошибки:**

   * При выполнении арифметических операций с числами, которые не могут быть представлены точно, возникает ошибка округления.
   * Например, 0.1 + 0.2 в двоичном представлении не равно точно 0.3, а приблизительно равно 0.30000000000000004.
   * При сравнении на равенство это может привести к неожиданным результатам.
3. **Рекомендации:**

   * Для финансовых расчетов или других задач, где требуется высокая точность, рекомендуется использовать модуль `decimal.Decimal`, который обеспечивает точное представление десятичных чисел.
   * При сравнении чисел с плавающей точкой на равенство следует использовать сравнение с допуском (epsilon): `abs(a - b) < 1e-9`.

### `nakoplenie_oshibok()`

**Описание:**

Демонстрирует накопление ошибок округления при многократном сложении чисел с плавающей точкой. При многократном выполнении операций с числами, которые не могут быть представлены точно, ошибки округления могут накапливаться и приводить к значительному отклонению результата от ожидаемого значения.

**Пример использования:**

```python
from matplobblib.nm.base import nakoplenie_oshibok

nakoplenie_oshibok()
```

**Вывод:**

```
накопление ошибок
100000.0
0.0
```

(Фактический вывод может быть, например: `100000.00000143294` и `-1.43294e-06`, или другие незначительные отклонения.)

**Теоретическая справка:**

1. **Накопление ошибок округления:**

   * Как и в случае с `oshibka_okrugleniya()`, число 0.1 не может быть точно представлено в двоичной системе.
   * При каждом сложении 0.1 с переменной `x` возникает небольшая ошибка округления.
   * После миллиона сложений эти ошибки накапливаются, и результат может отличаться от теоретического значения 100,000.0.
2. **Влияние порядка операций и реализации:**

   * Величина ошибки может зависеть от порядка выполнения операций и от реализации `float` в конкретном интерпретаторе Python.
3. **Альтернативы:**

   * Для точных вычислений можно использовать целочисленную арифметику (например, операции с копейками вместо рублей) или модуль `decimal.Decimal`.
   * В научных вычислениях применяются алгоритмы компенсированного суммирования, такие как алгоритм Кэхэна, для минимизации накопления ошибок.

### `poterya_znachimosti()`

**Описание:**

Демонстрирует потерю значимости при вычислении экспоненты малых чисел. Потеря значимости происходит при вычитании близких значений, когда результат может быть сильно искажен из-за ошибок округления.

**Пример использования:**

```python
from matplobblib.nm.base import poterya_znachimosti

poterya_znachimosti()
```

**Вывод:**

```
потеря значимости
1.0000000827.40371e-10
```

(Фактический результат может варьироваться в зависимости от точности вычислений NumPy.)

**Теоретическая справка:**

1. **Комбинация факторов:**

   * Проблема вызвана сочетанием ограниченной точности представления чисел с плавающей точкой (IEEE 754) и вычитанием близких значений.
   * Для малых `x` известно, что `exp(x)` ≈ 1 + x + x²/2! + x³/3! + ...
   * При `x` = 1 × 10`<sup>`-10`</sup>`, `exp(x)` ≈ 1.0000000001, и `exp(x) - 1` ≈ 1 × 10`<sup>`-10`</sup>`.
2. **Потеря относительной точности:**

   * При вычислении `exp(1e-10)` результат будет близок к 1, но с небольшой ошибкой округления.
   * Вычитание 1 из этого значения приводит к потере значимых разрядов, и результат может быть сильно искажен.
3. **Альтернативные подходы:**

   * Используйте функцию `np.expm1(x)` из NumPy, которая специально разработана для точного вычисления `exp(x) - 1` для малых `x`.
   * Разложение в ряд Тейлора также может быть использовано для более точного вычисления.
   * Для вычислений с произвольной точностью можно использовать специализированные библиотеки.

### `summirovanie_po_Kohanu(numbers)`

**Описание:**

Реализует алгоритм суммирования по Кохану, который минимизирует ошибки округления при сложении большого количества чисел с плавающей точкой. Алгоритм использует дополнительную переменную для отслеживания и компенсации ошибок округления.

**Параметры:**

* `numbers` (list или ndarray): Последовательность чисел (с плавающей точкой или целых), которые необходимо просуммировать.

**Возвращаемое значение:**

* `float`: Сумма всех чисел с учетом компенсации ошибок округления.

**Пример использования:**

```python
from matplobblib.nm.base import summirovanie_po_Kohanu
import numpy as np

result = summirovanie_po_Kohanu([0.1] * 1000000)
print(result)  # Вывод будет близок к 100000.0 с высокой точностью

result = summirovanie_po_Kohanu(np.arange(1, 10**5))
print(result)  # Вывод: 4999950000.0
```

**Теоретическая справка:**

1. **Алгоритм Кохана:**

   * Алгоритм Кохана использует переменную `correction` для хранения накопленной ошибки округления.
   * На каждом шаге алгоритма вычисляется разница между текущим числом и накопленной ошибкой (`y = num - correction`).
   * Затем `y` добавляется к текущей сумме (`temp = total + y`).
   * Ошибка округления, возникшая при сложении `total` и `y`, вычисляется как `(temp - total) - y` и сохраняется в `correction`.
   * Таким образом, на каждом шаге алгоритм корректирует текущую сумму на величину накопленной ошибки, что позволяет значительно уменьшить общую ошибку округления при суммировании большого количества чисел.
2. **Преимущества:**

   * Алгоритм Кохана обеспечивает более высокую точность суммирования по сравнению с наивным суммированием, особенно при работе с большим количеством чисел или числами с разным порядком величины.
3. **Ссылки:**

   * Кахан В. (1965). "Further remarks on reducing truncation errors." Numerische Mathematik 7 (1): 488.
   * Higham, N. J. (2002). Accuracy and Stability of Numerical Algorithms (2nd ed.). SIAM.
   * Википедия: https://ru.wikipedia.org/wiki/Суммирование_по_Кохану
