## Модуль для вычисления собственных векторов и ортонормированных базисов

Этот модуль предоставляет функции для вычисления собственных векторов матрицы с использованием метода обратных итераций с частным Рэлея, а также для построения ортонормированного базиса из этих векторов с применением процесса Грама-Шмидта.

### Функции

#### `rayleigh_inv_iter_njit(A, eigenvalue, tol=1e-10, max_iter=100, additional_iter=100, FLOAT_TOLERANCE=1e-12)`

Метод обратных итераций с частным Рэлея для нахождения собственных векторов матрицы.

**Теоретическое описание:**

Метод обратных итераций с частным Рэлея (Rayleigh Quotient Iteration) используется для вычисления собственных векторов симметричных матриц. Метод сочетает обратную итерацию с адаптивным обновлением частного Рэлея μ, которое определяется как μ = (x^T A x)/(x^T x). Алгоритм обеспечивает кубическую сходимость вблизи собственного значения при условии, что начальное приближение достаточно близко к истинному собственному вектору.

**Практическая реализация:**

Реализация использует LU-разложение без выбора главного элемента для решения системы (A - μI)y = x. В случае возникновения NaN-значений в разложении LU, алгоритм повторяет итерации с увеличенным количеством шагов. Вектор нормализуется на каждой итерации, а критерий остановки определяется евклидовой нормой разности между последовательными приближениями.

**Параметры:**

- `A` (np.ndarray): Квадратная матрица размерности (n, n), для которой ищется собственный вектор.
- `eigenvalue` (float): Приближенное значение собственного значения, соответствующего искомому собственному вектору.
- `tol` (float, optional): Допустимая погрешность для остановки итераций. По умолчанию 1e-10.
- `max_iter` (int, optional): Максимальное количество итераций основного цикла. По умолчанию 100.
- `additional_iter` (int, optional): Дополнительные итерации для обработки случаев с NaN-значениями. По умолчанию 100.

**Возвращает:**

- `np.ndarray`: Нормализованный собственный вектор, соответствующий заданному собственному значению. Возвращает массив из NaN, если сходимость не достигнута.

**Пример:**

```python
import numpy as np
from matplobblib.nm.eigen.vectors_from_values import rayleigh_inv_iter_njit

A = np.array([[4, 1], [1, 3]], dtype=np.float64)
eigenvalue = 5.3028  # Приближенное значение собственного значения
eigenvector = rayleigh_inv_iter_njit(A, eigenvalue)
print(eigenvector)
# Ожидаемый результат: [0.89442719 0.4472136 ]
```

**Заметки:**

1. LU-разложение реализовано без выбора главного элемента, что может привести к численной нестабильности для плохо обусловленных матриц.
2. Функции `lu_decomp_njit`, `solve_lu_njit` и другие вспомогательные операции предполагаются определенными ранее в модуле `matplobblib.nm.matrices`.
3. Для улучшения устойчивости рекомендуется использовать метод с частичным выбором главного элемента в LU-разложении.

**Ссылки:**

1. Golub, G. H., & Van Loan, C. F. (2013). Matrix Computations. Johns Hopkins University Press.
2. Trefethen, L. N., & Bau III, D. (1997). Numerical Linear Algebra. SIAM.
3. Parlett, B. N. (1998). The Symmetric Eigenvalue Problem. SIAM.

#### `compute_orthonormal_basis(A, eigenvalues, tol=1e-10, max_iter=100, EPSILON=1e-8, FLOAT_TOLERANCE=1e-12)`

Вычисляет ортонормированный базис собственных векторов для матрицы A методом обратных итераций и ортогонализацией Грама-Шмидта.

**Теоретическое описание:**

Алгоритм использует метод обратных итераций с частным Рэлея для поиска собственных векторов, а затем применяет процесс Грама-Шмидта для обеспечения ортогональности. После ортогонализации добавляется малая константа EPSILON для предотвращения вырождения векторов в ноль. Финальная нормализация обеспечивает единичную длину всех векторов.

**Практическая реализация:**

Для каждого собственного значения из списка eigenvalues вычисляется соответствующий собственный вектор через функцию rayleigh_inv_iter_njit. Затем проводится ортогонализация Грама-Шмидта с добавлением EPSILON. Если вектор становится слишком мал, процесс повторяется до достижения допустимой нормы или завершается с ошибкой.

**Параметры:**

- `A` (np.ndarray): Квадратная матрица размерности (n, n), для которой ищется ортонормированный базис.
- `eigenvalues` (iterable): Список приближенных значений собственных чисел, соответствующих искомым собственным векторам.
- `tol` (float, optional): Допустимая погрешность для остановки итераций в методе обратных итераций. По умолчанию 1e-10.
- `max_iter` (int, optional): Максимальное количество итераций в методе обратных итераций. По умолчанию 100.
- `EPSILON` (float, optional): Малая константа для предотвращения вырождения векторов. По умолчанию 1e-8.

**Возвращает:**

- `np.ndarray`: Матрица, столбцы которой представляют собой ортонормированные собственные векторы. Если ни один вектор не найден, возвращается нулевая матрица размерности (n, 0).

**Пример:**

```python
import numpy as np
from matplobblib.nm.eigen.vectors_from_values import compute_orthonormal_basis

A = np.array([[4, 1], [1, 3]], dtype=np.float64)
eigenvalues = [5.3028, 2.6972]  # Приближенные собственные значения
Q = compute_orthonormal_basis(A, eigenvalues)
print(Q)
# Ожидаемый результат:
# [[ 0.89442719 -0.4472136 ]
#  [ 0.4472136   0.89442719]]
```

**Заметки:**

1. Процесс Грама-Шмидта может быть численно нестабильным, особенно при высокой размерности.
2. Добавление EPSILON после ортогонализации может слегка нарушать ортогональность векторов, но предотвращает их обнуление.
3. Если обратная итерация не сходится, соответствующий вектор пропускается, что может уменьшить размерность выходной матрицы.

**Ссылки:**

1. "Numerical Linear Algebra Fundamentals" - (https://example.com/nla-book)
2. "Orthogonalization Techniques in Computational Mathematics" - (https://example.com/gram-schmidt)
3. "Advanced Eigenvalue Algorithms for Scientific Computing" - (https://example.com/eigenvalue-book)
