from typing import Literal, Union

from classiq._internals.enum_utils import StrEnum


class AnalyzerProviderVendor(StrEnum):
    IBM_QUANTUM = "IBM Quantum"
    AZURE_QUANTUM = "Azure Quantum"
    AMAZON_BRAKET = "Amazon Braket"


class ProviderVendor(StrEnum):
    CLASSIQ = "Classiq"
    IBM_QUANTUM = "IBM Quantum"
    AZURE_QUANTUM = "Azure Quantum"
    AMAZON_BRAKET = "Amazon Braket"
    IONQ = "IonQ"


class ProviderTypeVendor:
    CLASSIQ = Literal[ProviderVendor.CLASSIQ]
    IBM_QUANTUM = Literal[ProviderVendor.IBM_QUANTUM]
    AZURE_QUANTUM = Literal[ProviderVendor.AZURE_QUANTUM]
    AMAZON_BRAKET = Literal[ProviderVendor.AMAZON_BRAKET]
    IONQ = Literal[ProviderVendor.IONQ]


class ClassiqAerBackendNames(StrEnum):
    AER_SIMULATOR = "aer_simulator"
    AER_SIMULATOR_STATEVECTOR = "aer_simulator_statevector"
    AER_SIMULATOR_DENSITY_MATRIX = "aer_simulator_density_matrix"
    AER_SIMULATOR_MATRIX_PRODUCT_STATE = "aer_simulator_matrix_product_state"


class IonqBackendNames(StrEnum):
    SIMULATOR = "simulator"
    HARMONY = "qpu.harmony"
    ARIA = "qpu.aria-1"
    S11 = "qpu.s11"


class AzureQuantumBackendNames(StrEnum):
    IONQ_ARIA_1 = "ionq.qpu.aria-1"
    IONQ_ARIA_2 = "ionq.qpu.aria-2"
    IONQ_QPU = "ionq.qpu"
    IONQ_SIMULATOR = "ionq.simulator"
    MICROSOFT_ESTIMATOR = "microsoft.estimator"
    MICROSOFT_FULLSTATE_SIMULATOR = "microsoft.simulator.fullstate"
    RIGETTI_ASPEN3 = "rigetti.qpu.aspen-m-3"
    RIGETTI_SIMULATOR = "rigetti.sim.qvm"
    QCI_MACHINE1 = "qci.machine1"
    QCI_NOISY_SIMULATOR = "qci.simulator.noisy"
    QCI_SIMULATOR = "qci.simulator"
    QUANTINUUM_API_VALIDATOR1_1 = "quantinuum.sim.h1-1sc"
    QUANTINUUM_API_VALIDATOR1_2 = "quantinuum.sim.h1-2sc"
    QUANTINUUM_API_VALIDATOR2_1 = "quantinuum.sim.h2-1sc"
    QUANTINUUM_QPU1_1 = "quantinuum.qpu.h1-1"
    QUANTINUUM_QPU1_2 = "quantinuum.qpu.h1-2"
    QUANTINUUM_SIMULATOR1_1 = "quantinuum.sim.h1-1e"
    QUANTINUUM_SIMULATOR1_2 = "quantinuum.sim.h1-2e"
    QUANTINUUM_QPU2 = "quantinuum.qpu.h2-1"
    QUANTINUUM_SIMULATOR2 = "quantinuum.sim.h2-1e"


class AmazonBraketBackendNames(StrEnum):
    AMAZON_BRAKET_SV1 = "SV1"
    AMAZON_BRAKET_TN1 = "TN1"
    AMAZON_BRAKET_DM1 = "dm1"
    AMAZON_BRAKET_ASPEN_11 = "Aspen-11"
    AMAZON_BRAKET_M_1 = "Aspen-M-1"
    AMAZON_BRAKET_IONQ = "IonQ Device"
    AMAZON_BRAKET_LUCY = "Lucy"


# The IBM devices were taken from:
#   from qiskit.providers.fake_provider import FakeProvider
#   provider = FakeProvider()
#   backends_list = provider.backends()
#   # Using _normalize_backend_name from `ibm_hardware_graphs.py`
#   the_devices = [_normalize_backend_name(str(backend)) for backend in backends_list]
class IBMQHardwareNames(StrEnum):
    ALMADEN = "Almaden"
    ARMONK = "Armonk"
    ATHENS = "Athens"
    BELEM = "Belem"
    BOEBLINGEN = "Boeblingen"
    BOGOTA = "Bogota"
    BROOKLYN = "Brooklyn"
    BURLINGTON = "Burlington"
    CAIRO = "Cairo"
    CAMBRIDGE = "Cambridge"
    # CAMBRIDGEALTERNATIVEBASIS = "CambridgeAlternativeBasis"
    CASABLANCA = "Casablanca"
    ESSEX = "Essex"
    GUADALUPE = "Guadalupe"
    HANOI = "Hanoi"
    JAKARTA = "Jakarta"
    JOHANNESBURG = "Johannesburg"
    KOLKATA = "Kolkata"
    LAGOS = "Lagos"
    LIMA = "Lima"
    LONDON = "London"
    MANHATTAN = "Manhattan"
    MANILA = "Manila"
    MELBOURNE = "Melbourne"
    MONTREAL = "Montreal"
    MUMBAI = "Mumbai"
    NAIROBI = "Nairobi"
    OPENPULSE2Q = "OpenPulse_2Q"
    OPENPULSE3Q = "OpenPulse_3Q"
    OURENSE = "Ourense"
    PARIS = "Paris"
    POUGHKEEPSIE = "Poughkeepsie"
    QASM_SIMULATOR = "qasm_simulator"
    QUITO = "Quito"
    ROCHESTER = "Rochester"
    ROME = "Rome"
    RUESCHLIKON = "Rueschlikon"
    SANTIAGO = "Santiago"
    SINGAPORE = "Singapore"
    SYDNEY = "Sydney"
    TENERIFE = "Tenerife"
    TOKYO = "Tokyo"
    TORONTO = "Toronto"
    VALENCIA = "Valencia"
    VIGO = "Vigo"
    WASHINGTON = "Washington"
    YORKTOWN = "Yorktown"


class NvidiaBackendNames(StrEnum):
    SIMULATOR = "nvidia_state_vector_simulator"


AllClassiqBackendNames = Union[ClassiqAerBackendNames, NvidiaBackendNames]

EXACT_SIMULATORS = {
    IonqBackendNames.SIMULATOR,
    AzureQuantumBackendNames.IONQ_SIMULATOR,
    AzureQuantumBackendNames.MICROSOFT_FULLSTATE_SIMULATOR,
    AmazonBraketBackendNames.AMAZON_BRAKET_SV1,
    AmazonBraketBackendNames.AMAZON_BRAKET_TN1,
    AmazonBraketBackendNames.AMAZON_BRAKET_DM1,
    *ClassiqAerBackendNames,
    *NvidiaBackendNames,
}

AllIBMQBackendNames = IBMQHardwareNames


AllBackendsNameByVendor = Union[
    AllIBMQBackendNames,
    AzureQuantumBackendNames,
    AmazonBraketBackendNames,
    IonqBackendNames,
    NvidiaBackendNames,
]

AllBackendsNameEnums = [
    IBMQHardwareNames,
    AzureQuantumBackendNames,
    AmazonBraketBackendNames,
    IonqBackendNames,
    NvidiaBackendNames,
]
