"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunningControlStateMachine = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
class RunningControlStateMachine extends sfn.StateMachine {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            definitionBody: (() => {
                const initStateListDefinition = new sfn.Pass(scope, 'InitStateListDefinition', {
                    result: sfn.Result.fromObject([
                        { name: 'RUNNING', emoji: '☺', state: 'running' },
                        { name: 'STOPPED', emoji: '😴', state: 'stopped' },
                    ]),
                    resultPath: '$.definition.stateList',
                });
                const prepareTopicValue = new sfn.Pass(scope, 'PrepareTopicValue', {
                    resultPath: '$.prepare.topic.values',
                    parameters: {
                        emoji: sfn.JsonPath.arrayGetItem(sfn.JsonPath.stringAt('$.definition.stateList[?(@.state == $.Result.CurrentState)].emoji'), 0),
                        status: sfn.JsonPath.arrayGetItem(sfn.JsonPath.stringAt('$.definition.stateList[?(@.state == $.Result.CurrentState)].name'), 0),
                        account: sfn.JsonPath.arrayGetItem(sfn.JsonPath.stringSplit(sfn.JsonPath.stringAt('$.TargetResource'), ':'), 4), // account
                        region: sfn.JsonPath.arrayGetItem(sfn.JsonPath.stringSplit(sfn.JsonPath.stringAt('$.TargetResource'), ':'), 3), // region
                    },
                }).next(new sfn.Pass(scope, 'GenerateTopic', {
                    resultPath: '$.Generate.Topic',
                    parameters: {
                        Subject: sfn.JsonPath.format('{} [{}] AWS EC2 Instance {} Running Notification [{}][{}]', sfn.JsonPath.stringAt('$.prepare.topic.values.emoji'), sfn.JsonPath.stringAt('$.prepare.topic.values.status'), sfn.JsonPath.stringAt('$.Params.Mode'), sfn.JsonPath.stringAt('$.prepare.topic.values.account'), sfn.JsonPath.stringAt('$.prepare.topic.values.region')),
                        TextMessage: sfn.JsonPath.format('{}\n\nAccount : {}\nRegion : {}\nIdentifier : {}\nStatus : {}', sfn.JsonPath.format('The status of the EC2 instance changed to {} due to the schedule.', sfn.JsonPath.stringAt('$.prepare.topic.values.status')), sfn.JsonPath.stringAt('$.prepare.topic.values.account'), sfn.JsonPath.stringAt('$.prepare.topic.values.region'), sfn.JsonPath.stringAt('$.Target.Identifier'), sfn.JsonPath.stringAt('$.prepare.topic.values.status')),
                        // todo: next send slack
                        //            SlackJsonMessage: {
                        //              attachments: [
                        //                {
                        //                  color: '#36a64f',
                        //                  pretext: sfn.JsonPath.format('{} The status of the RDS {} changed to {} due to the schedule.',
                        //                    sfn.JsonPath.stringAt('$.prepare.topic.values.emoji'),
                        //                    sfn.JsonPath.stringAt('$.Result.target.type'),
                        //                    sfn.JsonPath.stringAt('$.prepare.topic.values.status'),
                        //                  ),
                        //                  fields: [
                        //                    {
                        //                      title: 'Account',
                        //                      value: sfn.JsonPath.stringAt('$.prepare.topic.values.account'),
                        //                      short: true,
                        //                    },
                        //                    {
                        //                      title: 'Region',
                        //                      value: sfn.JsonPath.stringAt('$.prepare.topic.values.region'),
                        //                      short: true,
                        //                    },
                        //                    {
                        //                      title: 'Type',
                        //                      value: sfn.JsonPath.stringAt('$.Result.target.type'),
                        //                      short: true,
                        //                    },
                        //                    {
                        //                      title: 'Identifier',
                        //                      value: sfn.JsonPath.stringAt('$.Result.target.identifier'),
                        //                      short: true,
                        //                    },
                        //                    {
                        //                      title: 'Status',
                        //                      value: sfn.JsonPath.stringAt('$.prepare.topic.values.status'),
                        //                      short: true,
                        //                    },
                        //                  ],
                        //                },
                        //              ],
                        //            },
                    },
                }).next(new tasks.SnsPublish(scope, 'SendNotification', {
                    topic: props.notificationTopic,
                    subject: sfn.JsonPath.stringAt('$.Generate.Topic.Subject'),
                    message: sfn.TaskInput.fromObject({
                        default: sfn.JsonPath.stringAt('$.Generate.Topic.TextMessage'),
                        email: sfn.JsonPath.stringAt('$.Generate.Topic.TextMessage'),
                        // lambda: sfn.JsonPath.jsonToString(sfn.JsonPath.objectAt('$.Generate.Topic.SlackJsonMessage')),
                    }),
                    messagePerSubscriptionType: true,
                    resultPath: '$.Result.SNS',
                }).next(new sfn.Succeed(scope, 'InstanceStatusChangeSucceed'))));
                // aws resourcegroupstaggingapi get-resources --resource-type-filters ec2:instance
                // 👇 Get EC2 Instance Resource from Tag
                const getTargetResources = new tasks.CallAwsService(scope, 'GetTargetResources', {
                    iamResources: ['*'],
                    iamAction: 'tag:GetResources',
                    service: 'resourcegroupstaggingapi',
                    action: 'getResources',
                    parameters: {
                        ResourceTypeFilters: [
                            'ec2:instance',
                        ],
                        TagFilters: [
                            {
                                Key: sfn.JsonPath.stringAt('$.Params.TagKey'),
                                Values: sfn.JsonPath.stringAt('$.Params.TagValues'),
                            },
                        ],
                    },
                    resultPath: '$.Result',
                    // $.Result.TargetResources
                    resultSelector: {
                        TargetResources: sfn.JsonPath.listAt('$..ResourceTagMappingList[*].ResourceARN'),
                    },
                });
                initStateListDefinition.next(getTargetResources);
                const getTargetLength = new sfn.Pass(scope, 'CalculateArrayLength', {
                    resultPath: '$.TargetResourceLength', // 中間結果を格納する場所
                    parameters: {
                        'Length.$': 'States.ArrayLength($.Result.TargetResources)', // 配列の長さを計算
                    },
                });
                getTargetResources.next(getTargetLength);
                // 👇 Do Stop instance
                const stopInstance = new tasks.CallAwsService(scope, 'StopInstance', {
                    iamResources: ['*'],
                    service: 'ec2',
                    action: 'stopInstances',
                    parameters: {
                        InstanceIds: sfn.JsonPath.array(sfn.JsonPath.stringAt('$.Target.Identifier')),
                    },
                    resultPath: '$.Result',
                    resultSelector: {
                        'CurrentState.$': '$.StoppingInstances[0].CurrentState.Name',
                    },
                });
                // 👇 Do Start instances
                const startInstance = new tasks.CallAwsService(scope, 'StartInstance', {
                    iamResources: ['*'],
                    service: 'ec2',
                    action: 'startInstances',
                    parameters: {
                        InstanceIds: sfn.JsonPath.array(sfn.JsonPath.stringAt('$.Target.Identifier')),
                    },
                    resultPath: '$.Result',
                    resultSelector: {
                        'CurrentState.$': '$.StartingInstances[0].CurrentState.Name',
                    },
                });
                // 👇 Describe instance
                const describeInstance = new tasks.CallAwsService(scope, 'DescribeInstance', {
                    iamResources: ['*'],
                    service: 'ec2',
                    action: 'describeInstances',
                    parameters: {
                        InstanceIds: sfn.JsonPath.array(sfn.JsonPath.stringAt('$.Target.Identifier')),
                    },
                    resultPath: '$.Result',
                    resultSelector: {
                        'CurrentState.$': '$.Reservations[0].Instances[0].State.Name',
                    },
                });
                const statusChangeWait = new sfn.Wait(scope, 'StatusChangeWait', {
                    time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(20)),
                });
                statusChangeWait.next(describeInstance);
                stopInstance.next(statusChangeWait);
                startInstance.next(statusChangeWait);
                const describeTypeChoice = new sfn.Choice(scope, 'StatusChoice')
                    // instance start on status.stopped
                    .when(sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Start'), sfn.Condition.stringEquals('$.Result.CurrentState', 'stopped')), startInstance)
                    // instance stop on status.running
                    .when(sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Stop'), sfn.Condition.stringEquals('$.Result.CurrentState', 'running')), stopInstance)
                    // status change succeed, // todo generate topic
                    .when(sfn.Condition.or(sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Start'), sfn.Condition.stringEquals('$.Result.CurrentState', 'running')), sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Stop'), sfn.Condition.stringEquals('$.Result.CurrentState', 'stopped'))), prepareTopicValue)
                    .when(
                // start & starting/configuring-enhanced-monitoring/backing-up or stop modifying/stopping
                sfn.Condition.or(sfn.Condition.and(sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Start'), sfn.Condition.or(sfn.Condition.stringEquals('$.Result.CurrentState', 'pending')))), sfn.Condition.and(sfn.Condition.and(sfn.Condition.stringEquals('$.Params.Mode', 'Stop'), sfn.Condition.or(sfn.Condition.stringEquals('$.Result.CurrentState', 'stopping'), sfn.Condition.stringEquals('$.Result.CurrentState', 'shutting-down'))))), statusChangeWait)
                    .otherwise(new sfn.Fail(scope, 'StatusFail', {
                    cause: 'instance status fail.',
                }));
                describeInstance.next(describeTypeChoice);
                const resourceStatusChangingMap = new sfn.Map(scope, 'ResourceProcessingMap', {
                    itemsPath: sfn.JsonPath.stringAt('$.Result.TargetResources'),
                    parameters: {
                        TargetResource: sfn.JsonPath.stringAt('$$.Map.Item.Value'),
                        Params: sfn.JsonPath.stringAt('$.Params'),
                        definition: sfn.JsonPath.stringAt('$.definition'),
                        // definition: sfn.JsonPath.stringAt('$.definition'),
                    },
                    maxConcurrency: 10,
                }).itemProcessor(new sfn.Pass(scope, 'GetIdentifier', {
                    resultPath: '$.Target',
                    parameters: {
                        // arn:aws:ec2:ap-northeast-1:123456789012:instance/i-0000000000aaaaa
                        Identifier: sfn.JsonPath.arrayGetItem(sfn.JsonPath.stringSplit(sfn.JsonPath.stringAt('$.TargetResource'), '/'), 1),
                    },
                }).next(describeInstance));
                const targetResourcesExistCheck = new sfn.Choice(scope, 'TargetResourcesExistCheck')
                    .when(sfn.Condition.numberGreaterThan('$.TargetResourceLength.Length', 0), 
                //sfn.Condition.numberGreaterThan(sfn.JsonPath.arrayLength('$.Result.TargetResources'), 0),
                resourceStatusChangingMap)
                    .otherwise(new sfn.Succeed(scope, 'TargetResourcesNotFound'));
                getTargetLength.next(targetResourcesExistCheck);
                return sfn.DefinitionBody.fromChainable(initStateListDefinition);
            })(),
        });
    }
}
exports.RunningControlStateMachine = RunningControlStateMachine;
//# sourceMappingURL=data:application/json;base64,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