"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const role_1 = require("./role");
/**
 * An IAM role that only gets attached to the construct tree once it gets used, not before
 *
 * This construct can be used to simplify logic in other constructs
 * which need to create a role but only if certain configurations occur
 * (such as when AutoScaling is configured). The role can be configured in one
 * place, but if it never gets used it doesn't get instantiated and will
 * not be synthesized or deployed.
 */
class LazyRole extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.statements = new Array();
        this.policies = new Array();
        this.managedPolicies = new Array();
    }
    export() {
        return this.instantiate().export();
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param permission The permission statement to add to the policy document
     */
    addToPolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
        else {
            this.statements.push(statement);
        }
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        if (this.role) {
            this.role.attachInlinePolicy(policy);
        }
        else {
            this.policies.push(policy);
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param arn The ARN of the managed policy to attach.
     */
    attachManagedPolicy(arn) {
        if (this.role) {
            this.role.attachManagedPolicy(arn);
        }
        else {
            this.managedPolicies.push(arn);
        }
    }
    /**
     * Returns the ARN of this role.
     */
    get roleArn() {
        return this.instantiate().roleArn;
    }
    get roleId() {
        return this.instantiate().roleId;
    }
    get roleName() {
        return this.instantiate().roleName;
    }
    /**
     * Returns a Principal object representing the ARN of this role.
     */
    get principal() {
        return this.instantiate().principal;
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        return this.instantiate().grant(identity, ...actions);
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.instantiate().grantPassRole(identity);
    }
    instantiate() {
        if (!this.role) {
            const role = new role_1.Role(this, 'Default', this.props);
            this.statements.forEach(role.addToPolicy.bind(role));
            this.policies.forEach(role.attachInlinePolicy.bind(role));
            this.managedPolicies.forEach(role.attachManagedPolicy.bind(role));
            this.role = role;
        }
        return this.role;
    }
}
exports.LazyRole = LazyRole;
//# sourceMappingURL=data:application/json;base64,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