import cdk = require('@aws-cdk/cdk');
export declare class PolicyDocument extends cdk.Token {
    private readonly baseDocument?;
    private statements;
    /**
     * Creates a new IAM policy document.
     * @param defaultDocument An IAM policy document to use as an initial
     * policy. All statements of this document will be copied in.
     */
    constructor(baseDocument?: any);
    resolve(_context: cdk.ResolveContext): any;
    readonly isEmpty: boolean;
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    readonly statementCount: number;
    addStatement(statement: PolicyStatement): PolicyDocument;
}
/**
 * Represents an IAM principal.
 */
export declare abstract class PolicyPrincipal {
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     */
    assumeRoleAction: string;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     */
    abstract policyFragment(): PrincipalPolicyFragment;
}
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
export declare class PrincipalPolicyFragment {
    readonly principalJson: {
        [key: string]: string[];
    };
    readonly conditions: {
        [key: string]: any;
    };
    constructor(principalJson: {
        [key: string]: string[];
    }, conditions?: {
        [key: string]: any;
    });
}
export declare class ArnPrincipal extends PolicyPrincipal {
    readonly arn: string;
    constructor(arn: string);
    policyFragment(): PrincipalPolicyFragment;
}
export declare class AccountPrincipal extends ArnPrincipal {
    readonly accountId: any;
    constructor(accountId: any);
}
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
export declare class ServicePrincipal extends PolicyPrincipal {
    readonly service: string;
    private readonly opts;
    constructor(service: string, opts?: ServicePrincipalOpts);
    policyFragment(): PrincipalPolicyFragment;
}
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
export declare class CanonicalUserPrincipal extends PolicyPrincipal {
    readonly canonicalUserId: string;
    constructor(canonicalUserId: string);
    policyFragment(): PrincipalPolicyFragment;
}
export declare class FederatedPrincipal extends PolicyPrincipal {
    readonly federated: string;
    readonly conditions: {
        [key: string]: any;
    };
    assumeRoleAction: string;
    constructor(federated: string, conditions: {
        [key: string]: any;
    }, assumeRoleAction?: string);
    policyFragment(): PrincipalPolicyFragment;
}
export declare class AccountRootPrincipal extends AccountPrincipal {
    constructor();
}
/**
 * A principal representing all identities in all accounts
 */
export declare class AnyPrincipal extends ArnPrincipal {
    constructor();
}
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
export declare class Anyone extends AnyPrincipal {
}
export declare class CompositePrincipal extends PolicyPrincipal {
    private readonly principals;
    constructor(principal: PolicyPrincipal, ...additionalPrincipals: PolicyPrincipal[]);
    addPrincipals(...principals: PolicyPrincipal[]): this;
    policyFragment(): PrincipalPolicyFragment;
}
/**
 * Represents a statement in an IAM policy document.
 */
export declare class PolicyStatement extends cdk.Token {
    private action;
    private principal;
    private resource;
    private condition;
    private effect?;
    private sid?;
    constructor(effect?: PolicyStatementEffect);
    addAction(action: string): PolicyStatement;
    addActions(...actions: string[]): PolicyStatement;
    /**
     * Indicates if this permission has a "Principal" section.
     */
    readonly hasPrincipal: boolean;
    addPrincipal(principal: PolicyPrincipal): this;
    addAwsPrincipal(arn: string): this;
    addAwsAccountPrincipal(accountId: string): this;
    addArnPrincipal(arn: string): this;
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param region  the region in which the service principal lives (defaults to the current stack's region).
     */
    addServicePrincipal(service: string, opts?: ServicePrincipalOpts): this;
    addFederatedPrincipal(federated: any, conditions: {
        [key: string]: any;
    }): this;
    addAccountRootPrincipal(): this;
    addCanonicalUserPrincipal(canonicalUserId: string): this;
    addAnyPrincipal(): this;
    addResource(arn: string): PolicyStatement;
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources(): PolicyStatement;
    addResources(...arns: string[]): PolicyStatement;
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    readonly hasResource: boolean;
    describe(sid: string): PolicyStatement;
    /**
     * Sets the permission effect to allow access to resources.
     */
    allow(): PolicyStatement;
    /**
     * Sets the permission effect to deny access to resources.
     */
    deny(): PolicyStatement;
    /**
     * Add a condition to the Policy
     */
    addCondition(key: string, value: any): PolicyStatement;
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions: {
        [key: string]: any;
    }): PolicyStatement;
    /**
     * Add a condition to the Policy.
     *
     * @deprecated For backwards compatibility. Use addCondition() instead.
     */
    setCondition(key: string, value: any): PolicyStatement;
    limitToAccount(accountId: string): PolicyStatement;
    resolve(_context: cdk.ResolveContext): any;
    toJson(): any;
}
export declare enum PolicyStatementEffect {
    Allow = "Allow",
    Deny = "Deny"
}
/**
 * Options for a service principal.
 */
export interface ServicePrincipalOpts {
    /**
     * The region in which the service is operating.
     *
     * @default the current Stack's region.
     */
    readonly region?: string;
}
