"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const region_info_1 = require("@aws-cdk/region-info");
class PolicyDocument extends cdk.Token {
    /**
     * Creates a new IAM policy document.
     * @param defaultDocument An IAM policy document to use as an initial
     * policy. All statements of this document will be copied in.
     */
    constructor(baseDocument) {
        super();
        this.baseDocument = baseDocument;
        this.statements = new Array();
    }
    resolve(_context) {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = this.baseDocument || {};
        doc.Statement = doc.Statement || [];
        doc.Version = doc.Version || '2012-10-17';
        doc.Statement = doc.Statement.concat(this.statements);
        return doc;
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    addStatement(statement) {
        this.statements.push(statement);
        return this;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Represents an IAM principal.
 */
class PolicyPrincipal {
    constructor() {
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
}
exports.PolicyPrincipal = PolicyPrincipal;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PolicyPrincipal {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PolicyPrincipal {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        });
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PolicyPrincipal {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PolicyPrincipal {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.accountId).toString());
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PolicyPrincipal {
    constructor(principal, ...additionalPrincipals) {
        super();
        this.principals = new Array();
        this.assumeRoleAction = principal.assumeRoleAction;
        this.addPrincipals(principal);
        this.addPrincipals(...additionalPrincipals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment();
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            mergePrincipal(principalJson, p.policyFragment().principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement extends cdk.Token {
    constructor(effect = PolicyStatementEffect.Allow) {
        super();
        this.action = new Array();
        this.principal = {};
        this.resource = new Array();
        this.condition = {};
        this.effect = effect;
    }
    //
    // Actions
    //
    addAction(action) {
        this.action.push(action);
        return this;
    }
    addActions(...actions) {
        actions.forEach(action => this.addAction(action));
        return this;
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0;
    }
    addPrincipal(principal) {
        const fragment = principal.policyFragment();
        mergePrincipal(this.principal, fragment.principalJson);
        this.addConditions(fragment.conditions);
        return this;
    }
    addAwsPrincipal(arn) {
        return this.addPrincipal(new ArnPrincipal(arn));
    }
    addAwsAccountPrincipal(accountId) {
        return this.addPrincipal(new AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        return this.addAwsPrincipal(arn);
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param region  the region in which the service principal lives (defaults to the current stack's region).
     */
    addServicePrincipal(service, opts) {
        return this.addPrincipal(new ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        return this.addPrincipal(new FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        return this.addPrincipal(new AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        return this.addPrincipal(new CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        return this.addPrincipal(new Anyone());
    }
    //
    // Resources
    //
    addResource(arn) {
        this.resource.push(arn);
        return this;
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        return this.addResource('*');
    }
    addResources(...arns) {
        arns.forEach(r => this.addResource(r));
        return this;
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    describe(sid) {
        this.sid = sid;
        return this;
    }
    //
    // Effect
    //
    /**
     * Sets the permission effect to allow access to resources.
     */
    allow() {
        this.effect = PolicyStatementEffect.Allow;
        return this;
    }
    /**
     * Sets the permission effect to deny access to resources.
     */
    deny() {
        this.effect = PolicyStatementEffect.Deny;
        return this;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
        return this;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
        return this;
    }
    /**
     * Add a condition to the Policy.
     *
     * @deprecated For backwards compatibility. Use addCondition() instead.
     */
    setCondition(key, value) {
        return this.addCondition(key, value);
    }
    limitToAccount(accountId) {
        return this.addCondition('StringEquals', new cdk.Token(() => {
            return { 'sts:ExternalId': accountId };
        }));
    }
    //
    // Serialization
    //
    resolve(_context) {
        return this.toJson();
    }
    toJson() {
        return {
            Action: _norm(this.action),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            Resource: _norm(this.resource),
            Sid: _norm(this.sid),
        };
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
}
exports.PolicyStatement = PolicyStatement;
var PolicyStatementEffect;
(function (PolicyStatementEffect) {
    PolicyStatementEffect["Allow"] = "Allow";
    PolicyStatementEffect["Deny"] = "Deny";
})(PolicyStatementEffect = exports.PolicyStatementEffect || (exports.PolicyStatementEffect = {}));
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        const value = source[key];
        if (!Array.isArray(value)) {
            throw new Error(`Principal value must be an array (it will be normalized later): ${value}`);
        }
        target[key].push(...value);
    }
    return target;
}
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken extends cdk.Token {
    constructor(fn) {
        super();
        this.fn = fn;
    }
    resolve(context) {
        return this.fn(context.scope.node.stack);
    }
}
class ServicePrincipalToken extends cdk.Token {
    constructor(service, opts) {
        super();
        this.service = service;
        this.opts = opts;
    }
    resolve(ctx) {
        const region = this.opts.region || ctx.scope.node.stack.region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, ctx.scope.node.stack.urlSuffix);
    }
}
//# sourceMappingURL=data:application/json;base64,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