import { Construct } from '@aws-cdk/cdk';
import { Group } from './group';
import { PolicyDocument, PolicyPrincipal, PolicyStatement } from './policy-document';
import { IRole } from './role';
import { User } from './user';
/**
 * A construct that represents an IAM principal, such as a user, group or role.
 */
export interface IPrincipal {
    /**
     * The IAM principal of this identity (i.e. AWS principal, service principal, etc).
     */
    readonly principal: PolicyPrincipal;
    /**
     * Adds an IAM statement to the default inline policy associated with this
     * principal. If a policy doesn't exist, it is created.
     */
    addToPolicy(statement: PolicyStatement): void;
    /**
     * Attaches an inline policy to this principal.
     * This is the same as calling `policy.addToXxx(principal)`.
     * @param policy The policy resource to attach to this principal.
     */
    attachInlinePolicy(policy: Policy): void;
    /**
     * Attaches a managed policy to this principal.
     * @param arn The ARN of the managed policy
     */
    attachManagedPolicy(arn: string): void;
}
/**
 * @deprecated Use IPrincipal
 */
export declare type IIdentityResource = IPrincipal;
export interface PolicyProps {
    /**
     * The name of the policy. If you specify multiple policies for an entity,
     * specify unique names. For example, if you specify a list of policies for
     * an IAM role, each policy must have a unique name.
     *
     * @default Uses the logical ID of the policy resource, which is ensured to
     *      be unique within the stack.
     */
    readonly policyName?: string;
    /**
     * Users to attach this policy to.
     * You can also use `attachToUser(user)` to attach this policy to a user.
     */
    readonly users?: User[];
    /**
     * Roles to attach this policy to.
     * You can also use `attachToRole(role)` to attach this policy to a role.
     */
    readonly roles?: IRole[];
    /**
     * Groups to attach this policy to.
     * You can also use `attachToGroup(group)` to attach this policy to a group.
     */
    readonly groups?: Group[];
    /**
     * Initial set of permissions to add to this policy document.
     * You can also use `addPermission(statement)` to add permissions later.
     */
    readonly statements?: PolicyStatement[];
}
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles,
 * or groups. For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 */
export declare class Policy extends Construct {
    /**
     * The policy document.
     */
    readonly document: PolicyDocument;
    /**
     * The name of this policy.
     */
    readonly policyName: string;
    private readonly roles;
    private readonly users;
    private readonly groups;
    constructor(scope: Construct, id: string, props?: PolicyProps);
    /**
     * Adds a statement to the policy document.
     */
    addStatement(statement: PolicyStatement): void;
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user: User): void;
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role: IRole): void;
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group: Group): void;
    protected validate(): string[];
}
