"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, props.externalId);
        this.managedPolicyArns = props.managedPolicyArns || [];
        validateMaxSessionDuration(props.maxSessionDurationSec);
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: util_1.undefinedIfEmpty(() => this.managedPolicyArns),
            policies: _flatten(props.inlinePolicies),
            path: props.path,
            roleName: props.roleName,
            maxSessionDuration: props.maxSessionDurationSec,
        });
        this.roleId = role.roleId;
        this.roleArn = role.roleArn;
        this.principal = new policy_document_1.ArnPrincipal(this.roleArn);
        this.roleName = role.roleName;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Import a role that already exists
     */
    static import(scope, id, props) {
        return new ImportedRole(scope, id, props);
    }
    export() {
        return {
            roleArn: new cdk_1.CfnOutput(this, 'RoleArn', { value: this.roleArn }).makeImportValue(),
            roleId: new cdk_1.CfnOutput(this, 'RoleId', { value: this.roleId }).makeImportValue()
        };
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param permission The permission statement to add to the policy document
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatement(statement);
    }
    /**
     * Attaches a managed policy to this role.
     * @param arn The ARN of the managed policy to attach.
     */
    attachManagedPolicy(arn) {
        this.managedPolicyArns.push(arn);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        if (!identity) {
            return;
        }
        identity.addToPolicy(new policy_document_1.PolicyStatement()
            .addResource(this.roleArn)
            .addActions(...actions));
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        this.grant(identity, 'iam:PassRole');
    }
}
exports.Role = Role;
function createAssumeRolePolicy(principal, externalId) {
    const statement = new policy_document_1.PolicyStatement();
    statement
        .addPrincipal(principal)
        .addAction(principal.assumeRoleAction);
    if (externalId !== undefined) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalId });
    }
    return new policy_document_1.PolicyDocument().addStatement(statement);
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
/**
 * A role that already exists
 */
class ImportedRole extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.roleArn = props.roleArn;
        this._roleId = props.roleId;
        this.principal = new policy_document_1.ArnPrincipal(this.roleArn);
    }
    get roleId() {
        if (!this._roleId) {
            throw new Error(`No roleId specified for imported role`);
        }
        return this._roleId;
    }
    get roleName() {
        return this.node.stack.parseArn(this.roleArn).resourceName;
    }
    export() {
        return this.props;
    }
    addToPolicy(_statement) {
        // FIXME: Add warning that we're ignoring this
    }
    attachInlinePolicy(_policy) {
        // FIXME: Add warning that we're ignoring this
    }
    attachManagedPolicy(_arn) {
        // FIXME: Add warning that we're ignoring this
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(_identity, ..._actions) {
        // FIXME: Add warning that we're ignoring this
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(_identity) {
        // FIXME: Add warning that we're ignoring this
    }
}
//# sourceMappingURL=data:application/json;base64,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