"use strict";
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const lib_2 = require("../lib");
module.exports = {
    'the Permission class is a programming model for iam'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAction('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResource('myQueue');
        p.addResource('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${new cdk_1.Token({ account: 'account' })}name`);
        p.limitToAccount('12221121221');
        test.deepEqual(stack.node.resolve(p), { Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
        test.done();
    },
    'the PolicyDocument class is a dom for iam policy documents'(test) {
        const stack = new cdk_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addAction('sqs:SendMessage');
        p1.addResource('*');
        const p2 = new lib_1.PolicyStatement();
        p2.deny();
        p2.addActions('cloudformation:CreateStack');
        doc.addStatement(p1);
        doc.addStatement(p2);
        test.deepEqual(stack.node.resolve(doc), {
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', Resource: '*' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' }]
        });
        test.done();
    },
    'A PolicyDocument can be initialized with an existing policy, which is merged upon serialization'(test) {
        const stack = new cdk_1.Stack();
        const base = {
            Version: 'Foo',
            Something: 123,
            Statement: [
                { Statement1: 1 },
                { Statement2: 2 }
            ]
        };
        const doc = new lib_1.PolicyDocument(base);
        doc.addStatement(new lib_1.PolicyStatement().addResource('resource').addAction('action'));
        test.deepEqual(stack.node.resolve(doc), { Version: 'Foo',
            Something: 123,
            Statement: [{ Statement1: 1 },
                { Statement2: 2 },
                { Effect: 'Allow', Action: 'action', Resource: 'resource' }] });
        test.done();
    },
    'Permission allows specifying multiple actions upon construction'(test) {
        const stack = new cdk_1.Stack();
        const perm = new lib_1.PolicyStatement().addResource('MyResource').addActions('Action1', 'Action2', 'Action3');
        test.deepEqual(stack.node.resolve(perm), {
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
        test.done();
    },
    'PolicyDoc resolves to undefined if there are no permissions'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        test.deepEqual(stack.node.resolve(p), undefined);
        test.done();
    },
    'canonicalUserPrincipal adds a principal to a policy with the passed canonical user id'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = "averysuperduperlongstringfor";
        p.addPrincipal(new lib_1.CanonicalUserPrincipal(canoncialUser));
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                CanonicalUser: canoncialUser
            }
        });
        test.done();
    },
    'addAccountRootPrincipal adds a principal with the current account root'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                AWS: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":iam::",
                            { Ref: "AWS::AccountId" },
                            ":root"
                        ]
                    ]
                }
            }
        });
        test.done();
    },
    'addFederatedPrincipal adds a Federated principal with the passed value'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal("com.amazon.cognito", { StringEquals: { key: 'value' } });
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                Federated: "com.amazon.cognito"
            },
            Condition: {
                StringEquals: { key: 'value' }
            }
        });
        test.done();
    },
    'addAwsAccountPrincipal can be used multiple times'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        test.deepEqual(stack.node.resolve(p), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] }
                ]
            }
        });
        test.done();
    },
    'hasResource': {
        'false if there are no resources'(test) {
            test.equal(new lib_1.PolicyStatement().hasResource, false, 'hasResource should be false for an empty permission');
            test.done();
        },
        'true if there is one resource'(test) {
            test.equal(new lib_1.PolicyStatement().addResource('one-resource').hasResource, true, 'hasResource is true when there is one resource');
            test.done();
        },
        'true for multiple resources'(test) {
            const p = new lib_1.PolicyStatement();
            p.addResource('r1');
            p.addResource('r2');
            test.equal(p.hasResource, true, 'hasResource is true when there are multiple resource');
            test.done();
        },
    },
    'hasPrincipal': {
        'false if there is no principal'(test) {
            test.equal(new lib_1.PolicyStatement().hasPrincipal, false);
            test.done();
        },
        'true if there is a principal'(test) {
            const p = new lib_1.PolicyStatement();
            p.addAwsPrincipal('bla');
            test.equal(p.hasPrincipal, true);
            test.done();
        }
    },
    'statementCount returns the number of statement in the policy document'(test) {
        const p = new lib_1.PolicyDocument();
        test.equal(p.statementCount, 0);
        p.addStatement(new lib_1.PolicyStatement());
        test.equal(p.statementCount, 1);
        p.addStatement(new lib_1.PolicyStatement());
        test.equal(p.statementCount, 2);
        test.done();
    },
    'the { AWS: "*" } principal is represented as `Anyone` or `AnyPrincipal`'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.Anyone()));
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.AnyPrincipal()));
        p.addStatement(new lib_1.PolicyStatement().addAnyPrincipal());
        test.deepEqual(stack.node.resolve(p), {
            Statement: [
                { Effect: 'Allow', Principal: '*' },
                { Effect: 'Allow', Principal: '*' },
                { Effect: 'Allow', Principal: '*' }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addAwsPrincipal/addArnPrincipal are the aliases'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        p.addStatement(new lib_1.PolicyStatement().addAwsPrincipal('111222-A'));
        p.addStatement(new lib_1.PolicyStatement().addArnPrincipal('111222-B'));
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_2.ArnPrincipal('111222-C')));
        test.deepEqual(stack.node.resolve(p), {
            Statement: [{
                    Effect: 'Allow', Principal: { AWS: '111222-A' }
                },
                { Effect: 'Allow', Principal: { AWS: '111222-B' } },
                { Effect: 'Allow', Principal: { AWS: '111222-C' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addCanonicalUserPrincipal can be used to add cannonical user principals'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        p.addStatement(new lib_1.PolicyStatement().addCanonicalUserPrincipal('cannonical-user-1'));
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.CanonicalUserPrincipal('cannonical-user-2')));
        test.deepEqual(stack.node.resolve(p), {
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addPrincipal correctly merges array in'(test) {
        const stack = new cdk_1.Stack();
        const arrayPrincipal = {
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: () => new lib_2.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
        };
        const s = new lib_1.PolicyStatement().addAccountRootPrincipal()
            .addPrincipal(arrayPrincipal);
        test.deepEqual(stack.node.resolve(s), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar'
                ]
            }
        });
        test.done();
    },
    // https://github.com/awslabs/aws-cdk/issues/1201
    'policy statements with multiple principal types can be created using multiple addPrincipal calls'(test) {
        const stack = new cdk_1.Stack();
        const s = new lib_1.PolicyStatement()
            .addAwsPrincipal('349494949494')
            .addServicePrincipal('test.service')
            .addResource('resource')
            .addAction('action');
        test.deepEqual(stack.node.resolve(s), {
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource'
        });
        test.done();
    },
    'Service principals': {
        'regional service principals resolve appropriately'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('codedeploy.amazonaws.com');
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'regional service principals resolve appropriately (with user-set region)'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'obscure service principals resolve to the user-provided value'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('test.service-principal.dev');
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' }
            });
            test.done();
        },
    },
    'CompositePrincipal can be used to represent a principal that has multiple types': {
        'with a single principal'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement().addPrincipal(p);
            test.deepEqual(stack.node.resolve(statement), { Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
            test.done();
        },
        'conditions are not allowed on individual principals of a composite'(test) {
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am'));
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('federated', { condition: 1 })), /Components of a CompositePrincipal must not have conditions/);
            test.done();
        },
        'principals and conditions are a big nice merge'(test) {
            const stack = new cdk_1.Stack();
            // add via ctor
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'), new lib_2.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_2.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement().addPrincipal(p);
            // add via policy statement
            statement.addAwsPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: 123 });
            test.deepEqual(stack.node.resolve(statement), {
                Condition: {
                    cond2: { boom: 123 }
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                }
            });
            test.done();
        },
        'cannot mix types of assumeRoleAction in a single composite'(test) {
            // GIVEN
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('fed', {}, 'sts:Boom')), /Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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