"use strict";
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
module.exports = {
    'fails when policy is empty'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synthesizeStack(stack.name), /Policy is empty/);
        test.done();
    },
    'policy with statements'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('sqs:SendMessage'));
        policy.addStatement(new lib_1.PolicyStatement().addResource('arn').addAction('sns:Subscribe'));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
        test.done();
    },
    'policy name can be omitted, in which case the logical id will be used'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('sqs:SendMessage'));
        policy.addStatement(new lib_1.PolicyStatement().addResource('arn').addAction('sns:Subscribe'));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'policy can be attached users, groups and roles and added permissions via props'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement().addResource('*').addAction('dynamodb:PutItem')],
        });
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
        test.done();
    },
    'idempotent if a principal (user/group/role) is attached twice'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatement(new lib_1.PolicyStatement().addAction('*').addResource('*'));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'users, groups, roles and permissions can be added using methods'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('dynamodb:GetItem'));
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'policy can be attached to users, groups or role via methods on the principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('*'));
        test.deepEqual(app.synthesizeStack(stack.name).template, { Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'fails if policy name is not unique within a user/group/role'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        test.throws(() => p2.attachToUser(user), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToGroup(group), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToRole(role), /A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
        test.done();
    },
    'fails if policy is not attached to a principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synthesizeStack(stack.name), /Policy must be attached to at least one principal: user, group or role/);
        test.done();
    },
    'generated policy name only uses the last 128 characters of the logical id'(test) {
        test.equal(util_1.generatePolicyName('Foo'), 'Foo');
        const logicalId50 = '[' + dup(50 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId50), logicalId50);
        const logicalId128 = '[' + dup(128 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId128), logicalId128);
        const withPrefix = 'PREFIX' + logicalId128;
        test.equal(util_1.generatePolicyName(withPrefix), logicalId128, 'ensure prefix is omitted');
        test.done();
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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