"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'default role'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com')
        });
        assert_1.expect(stack).toMatch({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'a role can grant PassRole permissions'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const role = new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.ServicePrincipal('henk.amazonaws.com') });
        const user = new lib_1.User(stack, 'User');
        // WHEN
        role.grantPassRole(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'can supply externalId'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalId: 'SomeSecret',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'policy is created automatically when permissions are added'(test) {
        const stack = new cdk_1.Stack();
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com')
        });
        test.ok(!('MyRoleDefaultPolicyA36BE1DD' in assert_1.SynthUtils.toCloudFormation(stack).Resources), 'initially created without a policy');
        role.addToPolicy(new lib_1.PolicyStatement().addResource('myresource').addAction('myaction'));
        test.ok(assert_1.SynthUtils.toCloudFormation(stack).Resources.MyRoleDefaultPolicyA36BE1DD, 'policy resource created');
        assert_1.expect(stack).toMatch({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } },
                MyRoleDefaultPolicyA36BE1DD: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'myaction', Effect: 'Allow', Resource: 'myresource' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyRoleDefaultPolicyA36BE1DD',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }] } } } });
        test.done();
    },
    'managed policy arns can be supplied upon initialization and also added later'(test) {
        const stack = new cdk_1.Stack();
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
            managedPolicyArns: ['managed1', 'managed2']
        });
        role.attachManagedPolicy('managed3');
        assert_1.expect(stack).toMatch({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: ['managed1', 'managed2', 'managed3'] } } } });
        test.done();
    },
    'federated principal can change AssumeRoleAction'(test) {
        const stack = new cdk_1.Stack();
        const cognitoPrincipal = new lib_1.FederatedPrincipal('foo', { StringEquals: { key: 'value' } }, 'sts:AssumeSomething');
        new lib_1.Role(stack, 'MyRole', { assumedBy: cognitoPrincipal });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Version: "2012-10-17",
                Statement: [
                    {
                        Principal: { Federated: "foo" },
                        Condition: {
                            StringEquals: { key: "value" }
                        },
                        Action: "sts:AssumeSomething",
                        Effect: "Allow",
                    }
                ],
            }
        }));
        test.done();
    },
    'maxSessionDuration': {
        'is not specified by default'(test) {
            const stack = new cdk_1.Stack();
            new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            assert_1.expect(stack).toMatch({
                Resources: {
                    MyRoleF48FFE04: {
                        Type: "AWS::IAM::Role",
                        Properties: {
                            AssumeRolePolicyDocument: {
                                Statement: [
                                    {
                                        Action: "sts:AssumeRole",
                                        Effect: "Allow",
                                        Principal: {
                                            Service: "sns.amazonaws.com"
                                        }
                                    }
                                ],
                                Version: "2012-10-17"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'can be used to specify the maximum session duration for assuming the role'(test) {
            const stack = new cdk_1.Stack();
            new lib_1.Role(stack, 'MyRole', { maxSessionDurationSec: 3700, assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                MaxSessionDuration: 3700
            }));
            test.done();
        },
        'must be between 3600 and 43200'(test) {
            const stack = new cdk_1.Stack();
            const assumedBy = new lib_1.ServicePrincipal('bla');
            new lib_1.Role(stack, 'MyRole1', { assumedBy, maxSessionDurationSec: 3600 });
            new lib_1.Role(stack, 'MyRole2', { assumedBy, maxSessionDurationSec: 43200 });
            const expected = (val) => `maxSessionDuration is set to ${val}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`;
            test.throws(() => new lib_1.Role(stack, 'MyRole3', { assumedBy, maxSessionDurationSec: 60 }), expected(60));
            test.throws(() => new lib_1.Role(stack, 'MyRole4', { assumedBy, maxSessionDurationSec: 3599 }), expected(3599));
            test.throws(() => new lib_1.Role(stack, 'MyRole5', { assumedBy, maxSessionDurationSec: 43201 }), expected(43201));
            test.done();
        }
    },
    'allow role with multiple principals'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.CompositePrincipal(new lib_1.ServicePrincipal('boom.amazonaws.test'), new lib_1.ArnPrincipal('1111111'))
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "boom.amazonaws.test",
                            AWS: "1111111"
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'import/export'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const myRole = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('boom.boom.boom')
        });
        // WHEN
        const exportedRole = myRole.export();
        const importedRole = lib_1.Role.import(stack, 'ImportedRole', exportedRole);
        // THEN
        test.deepEqual(stack.node.resolve(exportedRole), {
            roleArn: { 'Fn::ImportValue': 'Stack:MyRoleRoleArn3388B7E2' },
            roleId: { 'Fn::ImportValue': 'Stack:MyRoleRoleIdF7B258D8' }
        });
        test.deepEqual(stack.node.resolve(importedRole.roleArn), { 'Fn::ImportValue': 'Stack:MyRoleRoleArn3388B7E2' });
        test.deepEqual(stack.node.resolve(importedRole.roleId), { 'Fn::ImportValue': 'Stack:MyRoleRoleIdF7B258D8' });
        test.deepEqual(stack.node.resolve(importedRole.roleName), {
            'Fn::Select': [1, {
                    'Fn::Split': ['/', {
                            'Fn::Select': [5, {
                                    'Fn::Split': [':', {
                                            'Fn::ImportValue': 'Stack:MyRoleRoleArn3388B7E2'
                                        }]
                                }]
                        }]
                }]
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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