"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToApiGatewayToLambda = void 0;
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const aws_cloudfront_apigateway_1 = require("@aws-solutions-constructs/aws-cloudfront-apigateway");
class CloudFrontToApiGatewayToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the CloudFrontToApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.RegionalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps);
        this.apiGateway.methods.forEach((apiMethod) => {
            // Override the API Gateway Authorization Type from AWS_IAM to NONE
            const child = apiMethod.node.findChild('Resource');
            if (child.authorizationType === 'AWS_IAM') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
                child.cfnOptions.metadata = {
                    cfn_nag: {
                        rules_to_suppress: [{
                                id: 'W59',
                                reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                            }]
                    }
                };
            }
        });
        const apiCloudfront = new aws_cloudfront_apigateway_1.CloudFrontToApiGateway(this, 'CloudFrontToApiGateway', {
            existingApiGatewayObj: this.apiGateway,
            cloudFrontDistributionProps: props.cloudFrontDistributionProps,
            insertHttpSecurityHeaders: props.insertHttpSecurityHeaders
        });
        this.cloudFrontWebDistribution = apiCloudfront.cloudFrontWebDistribution;
        this.edgeLambdaFunctionVersion = apiCloudfront.edgeLambdaFunctionVersion;
        this.cloudFrontLoggingBucket = apiCloudfront.cloudFrontLoggingBucket;
    }
}
exports.CloudFrontToApiGatewayToLambda = CloudFrontToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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