"""Client for interacting with the Fewsats API"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['Client']

# %% ../nbs/00_core.ipynb 2
from fastcore.utils import *
import os
import httpx
from typing import Dict, Any, List

# %% ../nbs/00_core.ipynb 6
class Client:
    "Client for interacting with the Fewsats API"
    def __init__(self,
                 api_key: str = None, # The API key for the Fewsats account
                 base_url: str = "https://hub-5n97k.ondigitalocean.app"): # The Fewsats API base URL
        self.api_key = api_key or os.environ.get("FEWSATS_API_KEY")
        if not self.api_key:
            raise ValueError("The api_key client option must be set either by passing api_key to the client or by setting the FEWSATS_API_KEY environment variable")
        self.base_url = base_url
        self._httpx_client = httpx.Client()
        self._httpx_client.headers.update({"Authorization": f"Token {self.api_key}"})


# %% ../nbs/00_core.ipynb 9
@patch
def _request(self: Client, 
             method: str, # The HTTP method to use
             path: str, # The path to request
             **kwargs) -> Dict[str, Any]:
    "Makes an authenticated request to Fewsats API"
    url = f"{self.base_url}/{path}"
    return  self._httpx_client.request(method, url, **kwargs)

# %% ../nbs/00_core.ipynb 12
@patch
def me(self: Client):
    "Retrieve the user's info."
    r = self._request("GET", "v0/users/me")
    r.raise_for_status()
    return r.json()

# %% ../nbs/00_core.ipynb 15
@patch
def balance(self: Client):
    "Retrieve the balance of the user's wallet."
    r = self._request("GET", "v0/wallets")
    r.raise_for_status()
    return r.json()

# %% ../nbs/00_core.ipynb 18
@patch
def payment_methods(self: Client) -> List[Dict[str, Any]]:
    "Retrieve the user's payment methods, raises an exception for error status codes."
    r = self._request("GET", "v0/stripe/payment-methods")
    r.raise_for_status()
    return r.json()

# %% ../nbs/00_core.ipynb 22
@patch
def simulate_payment(self: Client,
                    amount: str): # The amount in USD cents
    "Simulates a purchase, raises an exception for error status codes."
    assert amount.isdigit()
    return self._request("POST", "v0/l402/preview/purchase/amount", json={"amount_usd": amount})


# %% ../nbs/00_core.ipynb 26
@patch
def _pay_ln(self: Client,
         ln_invoice: str, # The Lightning Network invoice to pay
         description: str, # Short payment description
         l402_url: str = ""): # L402 URL of the resource
     "Pay an invoice, raises an exception for error status codes."
     p = {"invoice": ln_invoice, "description": description, "l402_url": l402_url, "macaroon":""}
     return self._request("POST", "v0/l402/purchases/direct", json=p)  

# %% ../nbs/00_core.ipynb 30
@patch
def pay(self:Client,
        purl:str, # payment endpoint URL
        pct:str, # payment context token
        amount:int, # amount in cents
        balance:int, # balance
        currency:str, # currency
        description:str, # description
        offer_id:str, # offer id
        payment_methods:list[str], # payment methods
        title:str, # offer title
        type:str # offer type
) -> dict: # payment status response
    "POST payment request. Returns payment status response"
    return self._request("POST", "v0/l402/purchases/from-offer", json={
        "payment_request_url": purl,
        "payment_context_token": pct,
        "offer": {
            "offer_id": offer_id,
            "title": title,
            "description": description,
            "amount": amount,
            "type": type,
            "currency": currency,
            "balance": balance,
            "payment_methods": payment_methods,
        },
    })
