import dataclasses
import pathlib
import re
import typing

import yaml

from whocan import _errors


def _force_to_list(
        value: typing.Union[typing.List[str], str]
) -> typing.List[str]:
    """Force the value to a list."""
    if isinstance(value, str):
        return [value]
    return value


def _policy_check(raise_issue: bool, message: str):
    """Raise the error if raise_issue is true."""
    if raise_issue:
        raise _errors.PolicyYamlInvalidError(message)


def _validate_yaml(raw_yaml: typing.Any):
    """Validate the yaml or raise an error if it is invalid."""
    if not isinstance(raw_yaml, dict):
        raise _errors.PolicyYamlInvalidError(
            'Top level of policy must be a dictionary.'
        )
    _policy_check(
        'statements' not in raw_yaml,
        'Missing required field "statements".'
    )
    _policy_check(
        not isinstance(raw_yaml['statements'], list),
        '"statements" must be a list.'
    )

    for i, statement in enumerate(raw_yaml['statements']):
        required = ['effect', 'resources', 'actions']
        for r in required:
            _policy_check(
                r not in statement,
                f'Missing required field "statements[{i}].{r}".'
            )
        _policy_check(
            statement['effect'] not in {'allow', 'deny'},
            f'Missing required field "statements[{i}].effect"'
            ' must be "allow" or "deny".'
        )
        _policy_check(
            not isinstance(statement['resources'], (str, list)),
            f'Missing required field "statements[{i}].resources"'
            ' must be a string or list.'
        )
        _policy_check(
            not isinstance(statement['actions'], (str, list)),
            f'Missing required field "statements[{i}].actions"'
            ' must be a string or list.'
        )
        if isinstance(statement['resources'], list):
            _policy_check(
                any(not isinstance(r, str) for r in statement['resources']),
                f'All members of "statements[{i}].resources" must be strings.'
            )
        if isinstance(statement['actions'], list):
            _policy_check(
                any(not isinstance(a, str) for a in statement['actions']),
                f'All members of "statements[{i}].actions" must be strings.'
            )


def _form_regex(
        base: str,
        arguments: typing.Dict[str, str],
        strict: bool,
) -> str:
    """Form a regex from the given base value and arguments."""
    previous = 0
    processed = []
    for m in re.finditer(r'((?:\${\s*(\w+)\s*})|\*+)', base):
        if m.start() != previous:
            processed.append(re.escape(base[previous:m.start()]))
        if m.group(1) == '*':
            processed.append('[^/]*')
        if m.group(1).startswith('**'):
            processed.append('.*')
        if m.group(2):
            parameter = m.group(2)
            if parameter not in arguments and strict:
                raise _errors.PolicyEvaluationError(
                    f'"{parameter}" unknown variable.'
                )
            processed.append(str(arguments.get(parameter, '')))
        previous = m.end()
    processed.append(re.escape(base[previous:]))
    pattern = ''.join(processed)
    return f'^{pattern}$'


@dataclasses.dataclass
class Line:
    """A single resource or action."""

    raw_line: str
    arguments: typing.Dict[str, str]
    strict: bool = True

    def is_match(self, value: str) -> bool:
        """Determine if the given value is a match for the line."""
        if self.raw_line == '*':
            return True
        values = value.split(':')
        pieces = self.raw_line.split(':')
        if len(values) != len(pieces):
            return False
        for piece, incoming in zip(pieces, values):
            pattern = _form_regex(piece, self.arguments, self.strict)
            if not re.fullmatch(pattern, incoming):
                return False
        return True


@dataclasses.dataclass
class Statement:
    """A singular set of actions and resources."""

    effect: str
    resources: typing.List[str]
    actions: typing.List[str]

    def evaluate(
            self,
            action: str,
            resource: str,
            arguments: typing.Dict[str, str] = None,
    ) -> typing.Optional[str]:
        """
        Evaluate the statement to determine if it allows, denys, or has no
        effect on the specified resource and action.

        :param action:
            The action being taken on the specified resource.
        :param resource:
            The resource on which the action is being taken.
        :param arguments:
            Arguments to pass into the policy before determining if
            access is allowed.
        :return:
            Either "allow", "deny" or None.
        """
        action_lines = [Line(l, arguments) for l in self.actions]
        action_match = any(l.is_match(action) for l in action_lines)
        resource_lines = [Line(l, arguments) for l in self.resources]
        resource_match = any(l.is_match(resource) for l in resource_lines)
        return self.effect if action_match and resource_match else None


@dataclasses.dataclass
class Policy:
    """An policy defining resource access."""

    statements: typing.List[Statement]

    def is_allowed(
            self,
            action: str,
            resource: str,
            arguments: typing.Dict[str, str] = None
    ) -> bool:
        """
        Determine if the given policy allows the specified action on the
        specified resource.

        :param action:
            The action being taken on the specified resource.
        :param resource:
            The resource on which the action is being taken.
        :param arguments:
            Arguments to pass into the policy before determining if
            access is allowed.
        :return:
            Whether the action is allowed on the resource.
        """
        return 'allow' == self.evaluate(action, resource, arguments)

    def evaluate(
            self,
            action: str,
            resource: str,
            arguments: typing.Dict[str, str] = None
    ) -> typing.Optional[str]:
        """
        Evaluate the policy to determine if it allows, denys, or makes no
        comment on the specified resource and action.

        :param action:
            The action being taken on the specified resource.
        :param resource:
            The resource on which the action is being taken.
        :param arguments:
            Arguments to pass into the policy before determining if
            access is allowed.
        :return:
            Either "allow", "deny" or None.
        """
        evaluations = [
            statement.evaluate(action, resource, arguments)
            for statement in self.statements
        ]
        if any(v == 'deny' for v in evaluations):
            return 'deny'
        if any(v == 'allow' for v in evaluations):
            return 'allow'
        return None

    @classmethod
    def load(cls, path: pathlib.Path) -> "Policy":
        """Load the specified policy from yaml."""
        try:
            raw_yaml =  yaml.safe_load(path.read_text())
        except yaml.YAMLError:
            raise _errors.PolicyYamlInvalidError('Invalid policy yaml.')
        _validate_yaml(raw_yaml)
        statements = [
            Statement(
                statement['effect'],
                _force_to_list(statement['resources']),
                _force_to_list(statement['actions']),
            )
            for statement in raw_yaml['statements']
        ]
        return Policy(statements)

