"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const api = require("aws-cdk-lib/aws-apigateway");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    return new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
}
test('override cognito properties', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolProps = {
        userPoolName: 'test',
        userVerification: {}
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPool', {
        AdminCreateUserConfig: {
            AllowAdminCreateUserOnly: true
        },
        EmailVerificationMessage: "The verification code to your new account is {####}",
        EmailVerificationSubject: "Verify your new account",
        SmsVerificationMessage: "The verification code to your new account is {####}",
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test",
        VerificationMessageTemplate: {
            DefaultEmailOption: "CONFIRM_WITH_CODE",
            EmailMessage: "The verification code to your new account is {####}",
            EmailSubject: "Verify your new account",
            SmsMessage: "The verification code to your new account is {####}"
        }
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.apiGateway).toBeDefined();
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiGatewayAuthorizer).toBeDefined();
});
test('override cognito cognitoUserPoolClientProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolClientProps = {
        authFlows: {
            userSrp: true
        }
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolClientProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPoolClient', {
        ExplicitAuthFlows: [
            "ALLOW_USER_SRP_AUTH",
            "ALLOW_REFRESH_TOKEN_AUTH"
        ],
    });
});
test('Check for default Cognito Auth Type', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('override Auth Type to COGNITO', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.COGNITO
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Try to override Auth Type to NONE', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                operationName: 'foo-bar'
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps to support CORS', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultCorsPreflightOptions: {
                allowOrigins: api.Cors.ALL_ORIGINS,
                allowMethods: api.Cors.ALL_METHODS
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "NONE"
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "NONE"
    });
});
test('Override apiGatewayProps with proxy = false and add POST method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('POST');
    // Super important to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "POST",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Override apiGatewayProps with proxy = false and add OPTIONS method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('OPTIONS');
    // Mandatory to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "NONE",
    });
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Confirm CheckLambdaProps is being called', () => {
    const stack = new cdk.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        existingLambdaObj,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        }
    };
    const app = () => {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    };
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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