"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenIdConnectProvider = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class OpenIdConnectProvider extends core_1.Resource {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_OpenIdConnectProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
        this.openIdConnectProviderIssuer = core_1.Arn.extractResourceName(this.openIdConnectProviderArn, 'oidc-provider');
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        const resourceName = core_1.Arn.extractResourceName(openIdConnectProviderArn, 'oidc-provider');
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
                this.openIdConnectProviderIssuer = resourceName;
            }
        }
        return new Import(scope, id);
    }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
_a = JSII_RTTI_SYMBOL_1;
OpenIdConnectProvider[_a] = { fqn: "@aws-cdk/aws-iam.OpenIdConnectProvider", version: "1.157.0" };
//# sourceMappingURL=data:application/json;base64,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