#!/bin/bash
#SBATCH --partition=PARTITION
#SBATCH --nodes=1
#SBATCH --ntasks=1
#SBATCH --cpus-per-task=THREADS_VALUE
#SBATCH --mem=MEMORY
#SBATCH --time=TIME
#SBATCH --job-name=emtax
#SBATCH --output=WORKDIR_PATH/emtax_%j.out
#SBATCH --error=WORKDIR_PATH/emtax_%j.err

# emtax workflow script
# Generated by emtax

echo "Starting emtax workflow at $(date)"

# Set variables
export WORKDIR="WORKDIR_PATH"
export RAWDATA_DIR="RAWDATA_DIR_PATH"
export RESULTS_DIR="RESULTS_DIR_PATH"
export KRAKEN_DB="KRAKEN_DB_PATH"
export CORN_DB="CORN_DB_PATH"
export THREADS=THREADS_VALUE

# Check if conda is available
if command -v conda &> /dev/null; then
    echo "Conda is available"
    
    # Initialize conda for bash
    CONDA_BASE=$(conda info --base)
    source "${CONDA_BASE}/etc/profile.d/conda.sh"
    
    # Activate conda environment
    echo "Activating conda environment: emtax_env"
    conda activate emtax_env
else
    echo "WARNING: Conda not available, proceeding without environment activation"
fi

# Check network connectivity
if ping -c 1 8.8.8.8 &> /dev/null; then
    echo "Network is reachable, using online mode"
else
    echo "WARNING: Network is unreachable, using offline mode"
fi

# Change to workflow directory
cd "${WORKDIR}" || { echo "ERROR: Failed to change to workflow directory"; exit 1; }

echo "Processing samples from config file: ${WORKDIR}/config.yaml"

# Create output directories
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2"
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/2_DNA_Bracken"
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/5_DNA_Relative_Abundance_Matrix_Python"
mkdir -p "${WORKDIR}/Preprocessed_Data/fastp"
mkdir -p "${WORKDIR}/Logs/fastp"

# Process each sample
for SAMPLE in SAMPLES; do
    echo "Processing sample: $SAMPLE"
    
    # Step 1: Combine lanes for each sample
    echo "Combining lanes for ${SAMPLE}..."
    mkdir -p "${WORKDIR}/Preprocessed_Data/combined"
    
    # Find all R1 and R2 files for this sample
    R1_FILES=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_*_R1.fastq.gz" | sort)
    R2_FILES=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_*_R2.fastq.gz" | sort)
    
    # Combine lanes
    cat $R1_FILES > "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz"
    cat $R2_FILES > "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz"
    
    # Step 2: Run fastp on combined files
    echo "Running fastp for ${SAMPLE}..."
    if command -v fastp &> /dev/null; then
        mkdir -p "${WORKDIR}/Preprocessed_Data/fastp"
        mkdir -p "${WORKDIR}/Logs/fastp"
        
        fastp -i "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz" \
              -I "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz" \
              -o "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
              -O "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" \
              --json "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" \
              --html "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html" \
              --thread "${THREADS}" \
              --report_title "${SAMPLE} Quality Report" \
              > "${WORKDIR}/Logs/fastp/${SAMPLE}.log" 2>&1 || echo "WARNING: fastp failed for ${SAMPLE}"
        
        echo "Fastp reports generated for ${SAMPLE}:"
        ls -la "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html"
    else
        echo "WARNING: fastp not available, skipping quality trimming for ${SAMPLE}"
    fi
    
    # Step 3: Run dehost on fastp output
    echo "Running dehost for ${SAMPLE}..."
    if command -v bowtie2 &> /dev/null; then
        mkdir -p "${WORKDIR}/Preprocessed_Data/dehost"
        
        # Check if Bowtie2 index exists (either as direct path or in a subdirectory)
        if [ -e "${CORN_DB}.1.bt2" ] || [ -e "${CORN_DB}.1.bt2l" ] || [ -e "${CORN_DB}/corn_db.1.bt2" ] || [ -e "${CORN_DB}/corn_db.1.bt2l" ]; then
            echo "Bowtie2 index found, proceeding with dehosting..."
            
            # Determine the correct index path
            if [ -e "${CORN_DB}.1.bt2" ] || [ -e "${CORN_DB}.1.bt2l" ]; then
                INDEX_PATH="${CORN_DB}"
            else
                INDEX_PATH="${CORN_DB}/corn_db"
            fi
            echo "Using Bowtie2 index at: ${INDEX_PATH}"
            # Run bowtie2 directly to Preprocessed_Data directory with correct naming
            bowtie2 -p "${THREADS}" -x "${INDEX_PATH}" \
                    -1 "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
                    -2 "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" \
                    -S /dev/null \
                    --un-conc-gz "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R%.fastq.gz"
            
            # Check if dehosting was successful
            if [ ! -f "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz" ] || [ ! -f "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz" ]; then
                # Fallback: use fastp output as dehosted files
                cp "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
                cp "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
            fi
            echo "Dehosting completed for ${SAMPLE}"
        else
            echo "WARNING: Bowtie2 index not found at ${CORN_DB}, using fastp output instead"
            # Use fastp output as dehosted files
            ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
            ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
        fi
    else
        echo "WARNING: bowtie2 not available, using fastp output instead"
        # Use fastp output as dehosted files
        ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
        ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
    fi
    
    # Step 4: Run kraken2 if available
    if command -v kraken2 &> /dev/null; then
        echo "Running kraken2 for taxonomic classification for ${SAMPLE}..."
        kraken2 --db "${KRAKEN_DB}" \
                --threads "${THREADS}" \
                --paired \
                --output "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken" \
                --report "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                "${RAWDATA_DIR}/${SAMPLE}_L001_R1.fastq.gz" \
                "${RAWDATA_DIR}/${SAMPLE}_L001_R2.fastq.gz" || echo "WARNING: kraken2 failed for ${SAMPLE}"
    else
        echo "WARNING: kraken2 not available, skipping taxonomic classification for ${SAMPLE}"
    fi
    
    # Run bracken if available
    if command -v bracken &> /dev/null && [ -f "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" ]; then
        echo "Running bracken for abundance estimation for ${SAMPLE}..."
        bracken -d "${KRAKEN_DB}" \
                -i "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                -o "${RESULTS_DIR}/Taxonomic_Profiling/2_DNA_Bracken/${SAMPLE}.bracken" \
                -r 150 -l S -t 10 || echo "WARNING: bracken failed for ${SAMPLE}"
    else
        echo "WARNING: bracken not available or kraken2 report missing, skipping abundance estimation for ${SAMPLE}"
    fi
done

# Generate abundance matrix
echo "Generating abundance matrix..."
python "${WORKDIR}/scripts/create_abundance_matrix.py" \
    --input_files "${RESULTS_DIR}"/Taxonomic_Profiling/2_DNA_Bracken/*.bracken \
    --output "${RESULTS_DIR}/Taxonomic_Profiling/5_DNA_Relative_Abundance_Matrix_Python/abundance_matrix.csv" || \
    echo "WARNING: Failed to generate abundance matrix"

echo "emtax workflow finished at $(date)"
