#pragma once
#include <finance-enums/common.hpp>
#include <finance-enums/sector.hpp>
#include <finance-enums/helpers/exports.hpp>
#include <iostream>
#include <vector>

namespace finance_enums {
static std::unordered_map<Sector, std::vector<IndustryGroup>> SectorToIndustryGroup = {
  {Sector::Energy,                {IndustryGroup::Energy}   },
  {Sector::Materials,             {IndustryGroup::Materials}},
  {Sector::Industrials,
   {
     IndustryGroup::CapitalGoods,
     IndustryGroup::CommercialAndProfessionalServices,
     IndustryGroup::Transportation,
   }                                                        },
  {Sector::ConsumerDiscretionary,
   {
     IndustryGroup::AutomobilesAndComponents,
     IndustryGroup::ConsumerDurablesAndApparel,
     IndustryGroup::ConsumerServices,
     IndustryGroup::ConsumerDiscretionaryDistributionAndRetail,
   }                                                        },
  {Sector::ConsumerStaples,
   {
     IndustryGroup::ConsumerStaplesDistributionAndRetail,
     IndustryGroup::FoodBeverageAndTobacco,
     IndustryGroup::HouseholdAndPersonalProducts,
   }                                                        },
  {Sector::HealthCare,
   {
     IndustryGroup::HealthCareEquipmentAndServices,
     IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences,
   }                                                        },
  {Sector::Financials,
   {
     IndustryGroup::Banks,
     IndustryGroup::FinancialServices,
     IndustryGroup::Insurance,
   }                                                        },
  {Sector::InformationTechnology,
   {
     IndustryGroup::SoftwareAndServices,
     IndustryGroup::TechnologyHardwareAndEquipment,
     IndustryGroup::SemiconductorsAndSemiconductorEquipment,
   }                                                        },
  {Sector::CommunicationServices,
   {
     IndustryGroup::TelecommunicationServices,
     IndustryGroup::MediaAndEntertainment,
   }                                                        },
  {Sector::Utilities,             {IndustryGroup::Utilities}},
  {Sector::RealEstate,
   {
     IndustryGroup::EquityRealEstateInvestmentTrusts,
     IndustryGroup::RealEstateManagementAndDevelopment,
   }                                                        }
};

std::vector<IndustryGroup> inline Sector_to_IndustryGroup(Sector& x) { return SectorToIndustryGroup[x]; }

static std::unordered_map<IndustryGroup, Sector> IndustryGroupToSector = {
  {IndustryGroup::Energy,                                      Sector::Energy               },
  {IndustryGroup::Materials,                                   Sector::Materials            },
  {IndustryGroup::CapitalGoods,                                Sector::Industrials          },
  {IndustryGroup::CommercialAndProfessionalServices,           Sector::Industrials          },
  {IndustryGroup::Transportation,                              Sector::Industrials          },
  {IndustryGroup::AutomobilesAndComponents,                    Sector::ConsumerDiscretionary},
  {IndustryGroup::ConsumerDurablesAndApparel,                  Sector::ConsumerDiscretionary},
  {IndustryGroup::ConsumerServices,                            Sector::ConsumerDiscretionary},
  {IndustryGroup::ConsumerDiscretionaryDistributionAndRetail,  Sector::ConsumerDiscretionary},
  {IndustryGroup::ConsumerStaplesDistributionAndRetail,        Sector::ConsumerStaples      },
  {IndustryGroup::FoodBeverageAndTobacco,                      Sector::ConsumerStaples      },
  {IndustryGroup::HouseholdAndPersonalProducts,                Sector::ConsumerStaples      },
  {IndustryGroup::HealthCareEquipmentAndServices,              Sector::HealthCare           },
  {IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences, Sector::HealthCare           },
  {IndustryGroup::Banks,                                       Sector::Financials           },
  {IndustryGroup::FinancialServices,                           Sector::Financials           },
  {IndustryGroup::Insurance,                                   Sector::Financials           },
  {IndustryGroup::SoftwareAndServices,                         Sector::InformationTechnology},
  {IndustryGroup::TechnologyHardwareAndEquipment,              Sector::InformationTechnology},
  {IndustryGroup::SemiconductorsAndSemiconductorEquipment,     Sector::InformationTechnology},
  {IndustryGroup::TelecommunicationServices,                   Sector::CommunicationServices},
  {IndustryGroup::MediaAndEntertainment,                       Sector::CommunicationServices},
  {IndustryGroup::Utilities,                                   Sector::Utilities            },
  {IndustryGroup::EquityRealEstateInvestmentTrusts,            Sector::RealEstate           },
  {IndustryGroup::RealEstateManagementAndDevelopment,          Sector::RealEstate           }
};

Sector inline IndustryGroup_to_Sector(IndustryGroup& x) { return IndustryGroupToSector[x]; }

static std::unordered_map<IndustryGroup, std::vector<Industry>> IndustryGroupToIndustry = {
  {IndustryGroup::Energy,                                      {Industry::EnergyEquipmentAndServices, Industry::OilGasAndConsumeableFules}},
  {IndustryGroup::Materials,
   {
     Industry::Chemicals,
     Industry::ConstructionMaterials,
     Industry::ContainersAndPackaging,
     Industry::MetalsAndMining,
     Industry::PaperAndForestProducts,
   }                                                                                                                                      },
  {IndustryGroup::CapitalGoods,
   {
     Industry::AerospaceAndDefense,
     Industry::BuildingProducts,
     Industry::ConstructionAndEngineering,
     Industry::ElectricalEquipment,
     Industry::IndustrialConglomerates,
     Industry::Machinery,
     Industry::TradingCompaniesAndDistributors,
   }                                                                                                                                      },
  {IndustryGroup::CommercialAndProfessionalServices,
   {
     Industry::CommercialServicesAndSupplies,
     Industry::ProfessionalServices,
   }                                                                                                                                      },
  {IndustryGroup::Transportation,
   {
     Industry::AirFreightAndLogistics,
     Industry::PassengerAirlines,
     Industry::MarineTransportation,
     Industry::GroundTransportation,
     Industry::TransportationInfrastructure,
   }                                                                                                                                      },
  {IndustryGroup::AutomobilesAndComponents,                    {Industry::AutomobileComponents, Industry::Automobiles}                    },
  {IndustryGroup::ConsumerDurablesAndApparel,
   {
     Industry::HouseholdDurables,
     Industry::LeisureProducts,
     Industry::TextilesApparelAndLuxuryGoods,
   }                                                                                                                                      },
  {IndustryGroup::ConsumerServices,
   {
     Industry::HotelsRestaurantsAndLeisure,
     Industry::DiversifiedConsumerServices,
   }                                                                                                                                      },
  {IndustryGroup::ConsumerDiscretionaryDistributionAndRetail,
   {
     Industry::Distributors,
     Industry::BroadlineRetail,
     Industry::SpecialtyRetail,
   }                                                                                                                                      },
  {IndustryGroup::ConsumerStaplesDistributionAndRetail,        {Industry::ConsumerStaplesDistributionAndRetail}                           },
  {IndustryGroup::FoodBeverageAndTobacco,
   {
     Industry::Beverages,
     Industry::FoodProducts,
     Industry::Tobacco,
   }                                                                                                                                      },
  {IndustryGroup::HouseholdAndPersonalProducts,                {Industry::HouseholdProducts, Industry::PersonalCareProducts}              },
  {IndustryGroup::HealthCareEquipmentAndServices,
   {
     Industry::HealthCareEquipmentAndSupplies,
     Industry::HealthCareProvidersAndServices,
     Industry::HealthCareTechnology,
   }                                                                                                                                      },
  {IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences,
   {
     Industry::Biotechnology,
     Industry::Pharmaceuticals,
     Industry::LifeSciencesToolsAndServices,
   }                                                                                                                                      },
  {IndustryGroup::Banks,                                       {Industry::Banks}                                                          },
  {IndustryGroup::FinancialServices,
   {
     Industry::FinancialServices,
     Industry::ConsumerFinance,
     Industry::CapitalMarkets,
     Industry::MortgageRealEstateInvestmentTrusts,
   }                                                                                                                                      },
  {IndustryGroup::Insurance,                                   {Industry::Insurance}                                                      },
  {IndustryGroup::SoftwareAndServices,                         {Industry::ITServices, Industry::Software}                                 },
  {IndustryGroup::TechnologyHardwareAndEquipment,
   {
     Industry::CommunicationsEquipment,
     Industry::TechnologyHardwareStorageAndPeripherals,
     Industry::ElectronicEquipmentInstrumentsAndComponents,
   }                                                                                                                                      },
  {IndustryGroup::SemiconductorsAndSemiconductorEquipment,     {Industry::SemiconductorsAndSemiconductorEquipment}                        },
  {IndustryGroup::TelecommunicationServices,
   {
     Industry::DiversifiedTelecommunicationServices,
     Industry::WirelessTelecommunicationServices,
   }                                                                                                                                      },
  {IndustryGroup::MediaAndEntertainment,
   {
     Industry::Media,
     Industry::Entertainment,
     Industry::InteractiveMediaAndServices,
   }                                                                                                                                      },
  {IndustryGroup::Utilities,
   {
     Industry::ElectricUtilities,
     Industry::GasUtilities,
     Industry::MultiUtilities,
     Industry::WaterUtilities,
     Industry::IndependentPowerAndRenewableElectricityProducers,
   }                                                                                                                                      },
  {IndustryGroup::EquityRealEstateInvestmentTrusts,
   {
     Industry::DiversifiedREITs,
     Industry::IndustrialREITs,
     Industry::HotelAndResortREITs,
     Industry::OfficeREITs,
     Industry::HealthCareREITs,
     Industry::ResidentialREITs,
     Industry::RetailREITs,
     Industry::SpecializedREITs,
   }                                                                                                                                      },
  {IndustryGroup::RealEstateManagementAndDevelopment,          {Industry::RealEstateManagementAndDevelopment}                             }
};

std::vector<Industry> inline IndustryGroup_to_Industry(IndustryGroup& x) { return IndustryGroupToIndustry[x]; }

static std::unordered_map<Industry, IndustryGroup> IndustryToIndustryGroup = {
  {Industry::EnergyEquipmentAndServices,                       IndustryGroup::Energy                                     },
  {Industry::OilGasAndConsumeableFules,                        IndustryGroup::Energy                                     },
  {Industry::Chemicals,                                        IndustryGroup::Materials                                  },
  {Industry::ConstructionMaterials,                            IndustryGroup::Materials                                  },
  {Industry::ContainersAndPackaging,                           IndustryGroup::Materials                                  },
  {Industry::MetalsAndMining,                                  IndustryGroup::Materials                                  },
  {Industry::PaperAndForestProducts,                           IndustryGroup::Materials                                  },
  {Industry::AerospaceAndDefense,                              IndustryGroup::CapitalGoods                               },
  {Industry::BuildingProducts,                                 IndustryGroup::CapitalGoods                               },
  {Industry::ConstructionAndEngineering,                       IndustryGroup::CapitalGoods                               },
  {Industry::ElectricalEquipment,                              IndustryGroup::CapitalGoods                               },
  {Industry::IndustrialConglomerates,                          IndustryGroup::CapitalGoods                               },
  {Industry::Machinery,                                        IndustryGroup::CapitalGoods                               },
  {Industry::TradingCompaniesAndDistributors,                  IndustryGroup::CapitalGoods                               },
  {Industry::CommercialServicesAndSupplies,                    IndustryGroup::CommercialAndProfessionalServices          },
  {Industry::ProfessionalServices,                             IndustryGroup::CommercialAndProfessionalServices          },
  {Industry::AirFreightAndLogistics,                           IndustryGroup::Transportation                             },
  {Industry::PassengerAirlines,                                IndustryGroup::Transportation                             },
  {Industry::MarineTransportation,                             IndustryGroup::Transportation                             },
  {Industry::GroundTransportation,                             IndustryGroup::Transportation                             },
  {Industry::TransportationInfrastructure,                     IndustryGroup::Transportation                             },
  {Industry::AutomobileComponents,                             IndustryGroup::AutomobilesAndComponents                   },
  {Industry::Automobiles,                                      IndustryGroup::AutomobilesAndComponents                   },
  {Industry::HouseholdDurables,                                IndustryGroup::ConsumerDurablesAndApparel                 },
  {Industry::LeisureProducts,                                  IndustryGroup::ConsumerDurablesAndApparel                 },
  {Industry::TextilesApparelAndLuxuryGoods,                    IndustryGroup::ConsumerDurablesAndApparel                 },
  {Industry::HotelsRestaurantsAndLeisure,                      IndustryGroup::ConsumerServices                           },
  {Industry::DiversifiedConsumerServices,                      IndustryGroup::ConsumerServices                           },
  {Industry::Distributors,                                     IndustryGroup::ConsumerDiscretionaryDistributionAndRetail },
  {Industry::BroadlineRetail,                                  IndustryGroup::ConsumerDiscretionaryDistributionAndRetail },
  {Industry::SpecialtyRetail,                                  IndustryGroup::ConsumerDiscretionaryDistributionAndRetail },
  {Industry::ConsumerStaplesDistributionAndRetail,             IndustryGroup::ConsumerStaplesDistributionAndRetail       },
  {Industry::Beverages,                                        IndustryGroup::FoodBeverageAndTobacco                     },
  {Industry::FoodProducts,                                     IndustryGroup::FoodBeverageAndTobacco                     },
  {Industry::Tobacco,                                          IndustryGroup::FoodBeverageAndTobacco                     },
  {Industry::HouseholdProducts,                                IndustryGroup::HouseholdAndPersonalProducts               },
  {Industry::PersonalCareProducts,                             IndustryGroup::HouseholdAndPersonalProducts               },
  {Industry::HealthCareEquipmentAndSupplies,                   IndustryGroup::HealthCareEquipmentAndServices             },
  {Industry::HealthCareProvidersAndServices,                   IndustryGroup::HealthCareEquipmentAndServices             },
  {Industry::HealthCareTechnology,                             IndustryGroup::HealthCareEquipmentAndServices             },
  {Industry::Biotechnology,                                    IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences},
  {Industry::Pharmaceuticals,                                  IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences},
  {Industry::LifeSciencesToolsAndServices,                     IndustryGroup::PharmaceuticalsBiotechnologyAndLifeSciences},
  {Industry::Banks,                                            IndustryGroup::Banks                                      },
  {Industry::FinancialServices,                                IndustryGroup::FinancialServices                          },
  {Industry::ConsumerFinance,                                  IndustryGroup::FinancialServices                          },
  {Industry::CapitalMarkets,                                   IndustryGroup::FinancialServices                          },
  {Industry::MortgageRealEstateInvestmentTrusts,               IndustryGroup::FinancialServices                          },
  {Industry::Insurance,                                        IndustryGroup::Insurance                                  },
  {Industry::ITServices,                                       IndustryGroup::SoftwareAndServices                        },
  {Industry::Software,                                         IndustryGroup::SoftwareAndServices                        },
  {Industry::CommunicationsEquipment,                          IndustryGroup::TechnologyHardwareAndEquipment             },
  {Industry::TechnologyHardwareStorageAndPeripherals,          IndustryGroup::TechnologyHardwareAndEquipment             },
  {Industry::ElectronicEquipmentInstrumentsAndComponents,      IndustryGroup::TechnologyHardwareAndEquipment             },
  {Industry::SemiconductorsAndSemiconductorEquipment,          IndustryGroup::SemiconductorsAndSemiconductorEquipment    },
  {Industry::DiversifiedTelecommunicationServices,             IndustryGroup::TelecommunicationServices                  },
  {Industry::WirelessTelecommunicationServices,                IndustryGroup::TelecommunicationServices                  },
  {Industry::Media,                                            IndustryGroup::MediaAndEntertainment                      },
  {Industry::Entertainment,                                    IndustryGroup::MediaAndEntertainment                      },
  {Industry::InteractiveMediaAndServices,                      IndustryGroup::MediaAndEntertainment                      },
  {Industry::ElectricUtilities,                                IndustryGroup::Utilities                                  },
  {Industry::GasUtilities,                                     IndustryGroup::Utilities                                  },
  {Industry::MultiUtilities,                                   IndustryGroup::Utilities                                  },
  {Industry::WaterUtilities,                                   IndustryGroup::Utilities                                  },
  {Industry::IndependentPowerAndRenewableElectricityProducers, IndustryGroup::Utilities                                  },
  {Industry::DiversifiedREITs,                                 IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::IndustrialREITs,                                  IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::HotelAndResortREITs,                              IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::OfficeREITs,                                      IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::HealthCareREITs,                                  IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::ResidentialREITs,                                 IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::RetailREITs,                                      IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::SpecializedREITs,                                 IndustryGroup::EquityRealEstateInvestmentTrusts           },
  {Industry::RealEstateManagementAndDevelopment,               IndustryGroup::RealEstateManagementAndDevelopment         }
};

IndustryGroup inline Industry_to_IndustryGroup(Industry& x) { return IndustryToIndustryGroup[x]; }

static std::unordered_map<Industry, std::vector<SubIndustry>> IndustryToSubIndustries = {
  {Industry::EnergyEquipmentAndServices,                       {SubIndustry::OilAndGasDrilling, SubIndustry::OilAndGasEquipmentAndServices}               },
  {Industry::OilGasAndConsumeableFules,
   {SubIndustry::IntegratedOilAndGas, SubIndustry::OilAndGasExplorationAndProduction,
    SubIndustry::OilAndGasRefiningAndMarketing, SubIndustry::OilAndGasStorageAndTransportation,
    SubIndustry::CoalAndConsumeableFuels}                                                                                                                 },
  {Industry::Chemicals,
   {SubIndustry::CommodityChemicals, SubIndustry::DiversifiedChemicals,
    SubIndustry::FertilizersAndAgriculturalChemicals, SubIndustry::IndustrialGases, SubIndustry::SpecialtyChemicals}                                      },
  {Industry::ConstructionMaterials,                            {SubIndustry::ConstructionMaterials}                                                       },
  {Industry::ContainersAndPackaging,
   {SubIndustry::MetalGlassAndPlasticContainers, SubIndustry::PaperAndPlasticPackagingProductsAndMaterials}                                               },
  {Industry::MetalsAndMining,
   {SubIndustry::Aluminum, SubIndustry::DiversifiedMetalsAndMining, SubIndustry::Copper, SubIndustry::Gold,
    SubIndustry::PreciousMetalsAndMinerals, SubIndustry::Silver, SubIndustry::Steel}                                                                      },
  {Industry::PaperAndForestProducts,                           {SubIndustry::ForestProducts, SubIndustry::PaperProducts}                                  },
  {Industry::AerospaceAndDefense,                              {SubIndustry::AerospaceAndDefense}                                                         },
  {Industry::BuildingProducts,                                 {SubIndustry::BuildingProducts}                                                            },
  {Industry::ConstructionAndEngineering,                       {SubIndustry::ConstructionAndEngineering}                                                  },
  {Industry::ElectricalEquipment,
   {SubIndustry::ElectricalComponentsAndEquipment, SubIndustry::HeavyElectricalEquipment}                                                                 },
  {Industry::IndustrialConglomerates,                          {SubIndustry::IndustrialConglomerates}                                                     },
  {Industry::Machinery,
   {SubIndustry::ConstructionMachineryAndHeavyTransportationEquipment, SubIndustry::AgriculturalAndFarmMachinery,
    SubIndustry::IndustrialMachinerySuppliesAndComponents}                                                                                                },
  {Industry::TradingCompaniesAndDistributors,                  {SubIndustry::TradingCompaniesAndDistributors}                                             },
  {Industry::CommercialServicesAndSupplies,
   {SubIndustry::CommercialPrinting, SubIndustry::EnvironmentalAndFacilitiesServices,
    SubIndustry::OfficeServicesAndSupplies, SubIndustry::DiviersifiedSupportServices,
    SubIndustry::SecurityAndAlarmServices}                                                                                                                },
  {Industry::ProfessionalServices,
   {SubIndustry::HumanResourcesAndEmploymentServices, SubIndustry::ResearchAndConsultingServices,
    SubIndustry::DataProcessingAndOutsourcedServices}                                                                                                     },
  {Industry::AirFreightAndLogistics,                           {SubIndustry::AirFreightAndLogistics}                                                      },
  {Industry::PassengerAirlines,                                {SubIndustry::PassengerAirlines}                                                           },
  {Industry::MarineTransportation,                             {SubIndustry::MarineTransportation}                                                        },
  {Industry::GroundTransportation,
   {SubIndustry::RailTransportation, SubIndustry::CargoGroundTransportation,
    SubIndustry::PassengerGroundTransportation}                                                                                                           },
  {Industry::TransportationInfrastructure,
   {SubIndustry::AirportServices, SubIndustry::HighwaysAndRailtracks, SubIndustry::MarinePortsAndServices}                                                },
  {Industry::AutomobileComponents,                             {SubIndustry::AutomotivePartsAndEquipment, SubIndustry::TiresAndRubber}                    },
  {Industry::Automobiles,                                      {SubIndustry::AutomobileManufacturers, SubIndustry::MotorcycleManufacturers}               },
  {Industry::HouseholdDurables,
   {SubIndustry::ConsumerElectronics, SubIndustry::HomeFurnishings, SubIndustry::Homebuilding,
    SubIndustry::HouseholdAppliances, SubIndustry::HousewaresAndSpecialties}                                                                              },
  {Industry::LeisureProducts,                                  {SubIndustry::LeisureProducts}                                                             },
  {Industry::TextilesApparelAndLuxuryGoods,
   {SubIndustry::ApparelAccessoriesAndLuxuryGoods, SubIndustry::Footwear, SubIndustry::Textiles}                                                          },
  {Industry::HotelsRestaurantsAndLeisure,
   {SubIndustry::CasinosAndGaming, SubIndustry::HotelsResortsAndCruiseLines, SubIndustry::LeisureFacilities,
    SubIndustry::Restaurants}                                                                                                                             },
  {Industry::DiversifiedConsumerServices,                      {SubIndustry::EducationServices, SubIndustry::SpecializedConsumerServices}                 },
  {Industry::Distributors,                                     {SubIndustry::Distributors}                                                                },
  {Industry::BroadlineRetail,                                  {SubIndustry::BroadlineRetail}                                                             },
  {Industry::SpecialtyRetail,
   {SubIndustry::ApparelRetail, SubIndustry::ComputerAndElectronicsretail, SubIndustry::HomeImprovementRetail,
    SubIndustry::OtherSpecialtyRetail, SubIndustry::AutomotiveRetail, SubIndustry::HomefurnishingRetail}                                                  },
  {Industry::ConsumerStaplesDistributionAndRetail,
   {SubIndustry::DrugRetail, SubIndustry::FoodDistributors, SubIndustry::FoodRetail,
    SubIndustry::ConsumerStaplesMerchandiseRetail}                                                                                                        },
  {Industry::Beverages,
   {SubIndustry::Brewers, SubIndustry::DistillersAndVintners, SubIndustry::SoftDrinksAndNonAlcoholicBeverages}                                            },
  {Industry::FoodProducts,                                     {SubIndustry::AgriculturalProductsAndServices, SubIndustry::PackagedFoodsAndMeats}         },
  {Industry::Tobacco,                                          {SubIndustry::Tobacco}                                                                     },
  {Industry::HouseholdProducts,                                {SubIndustry::HouseholdProducts}                                                           },
  {Industry::PersonalCareProducts,                             {SubIndustry::PersonalCareProducts}                                                        },
  {Industry::HealthCareEquipmentAndSupplies,                   {SubIndustry::HealthCareEquipment, SubIndustry::HealthCareSupplies}                        },
  {Industry::HealthCareProvidersAndServices,
   {SubIndustry::HealthCareDistributors, SubIndustry::HealthCareServices, SubIndustry::HealthCareFacilities,
    SubIndustry::ManagedHealthCare}                                                                                                                       },
  {Industry::HealthCareTechnology,                             {SubIndustry::HealthCareTechnology}                                                        },
  {Industry::Biotechnology,                                    {SubIndustry::Biotechnology}                                                               },
  {Industry::Pharmaceuticals,                                  {SubIndustry::Pharmaceuticals}                                                             },
  {Industry::LifeSciencesToolsAndServices,                     {SubIndustry::LifeSciencesToolsAndServices}                                                },
  {Industry::Banks,                                            {SubIndustry::DiversifiedBanks, SubIndustry::RegionalBanks}                                },
  {Industry::FinancialServices,
   {SubIndustry::DiversifiedFinancialServices, SubIndustry::MultiSectorHoldings, SubIndustry::SpecializedFinance,
    SubIndustry::CommercialAndResidentialMortgageFinance, SubIndustry::TransactionAndPaymentProcessingServices}                                           },
  {Industry::ConsumerFinance,                                  {SubIndustry::ConsumerFinance}                                                             },
  {Industry::CapitalMarkets,
   {SubIndustry::AssetManagementAndCustodyBanks, SubIndustry::InvestmentBankingAndBrokerage,
    SubIndustry::DiversifiedCapitalMarkets, SubIndustry::FinancialExchangesAndData}                                                                       },
  {Industry::MortgageRealEstateInvestmentTrusts,               {SubIndustry::MortgageRealEstateInvestmentTrusts}                                          },
  {Industry::Insurance,
   {SubIndustry::InsuranceBrokers, SubIndustry::LifeAndHealthInsurance, SubIndustry::MultilineInsurance,
    SubIndustry::PropertyAndCasualtyInsurance, SubIndustry::Reinsurance}                                                                                  },
  {Industry::ITServices,                                       {SubIndustry::ITConsultingAndOtherServices, SubIndustry::InternetServicesAndInfrastructure}},
  {Industry::Software,                                         {SubIndustry::ApplicationSoftware, SubIndustry::SystemsSoftware}                           },
  {Industry::CommunicationsEquipment,                          {SubIndustry::CommunicationsEquipment}                                                     },
  {Industry::TechnologyHardwareStorageAndPeripherals,          {SubIndustry::TechnologyHardwareStorageAndPeripherals}                                     },
  {Industry::ElectronicEquipmentInstrumentsAndComponents,
   {
     SubIndustry::ElectronicEquipmentAndInstruments,
     SubIndustry::ElectronicComponents,
     SubIndustry::ElectronicManufacturingServices,
     SubIndustry::TechnologyDistributors,
   }                                                                                                                                                      },
  {Industry::SemiconductorsAndSemiconductorEquipment,
   {SubIndustry::SemiconductorMaterialsAndEquipment, SubIndustry::Semiconductors}                                                                         },
  {Industry::DiversifiedTelecommunicationServices,
   {SubIndustry::AlternativeCarriers, SubIndustry::IntegratedTelecommunicationServices}                                                                   },
  {Industry::WirelessTelecommunicationServices,                {SubIndustry::WirelessTelecommunicationServices}                                           },
  {Industry::Media,
   {SubIndustry::Advertising, SubIndustry::Broadcasting, SubIndustry::CableAndSatellite, SubIndustry::Publishing}                                         },
  {Industry::Entertainment,                                    {SubIndustry::MoviesAndEntertainment, SubIndustry::InteractiveHomeEntertainment}           },
  {Industry::InteractiveMediaAndServices,                      {SubIndustry::InteractiveMediaAndServices}                                                 },
  {Industry::ElectricUtilities,                                {SubIndustry::ElectricUtilities}                                                           },
  {Industry::GasUtilities,                                     {SubIndustry::GasUtilities}                                                                },
  {Industry::MultiUtilities,                                   {SubIndustry::MultiUtilities}                                                              },
  {Industry::WaterUtilities,                                   {SubIndustry::WaterUtilities}                                                              },
  {Industry::IndependentPowerAndRenewableElectricityProducers,
   {
     SubIndustry::IndependentPowerProducersAndEnergyTraders,
     SubIndustry::RenewableElectricity,
   }                                                                                                                                                      },
  {Industry::DiversifiedREITs,                                 {SubIndustry::DiversifiedREITs}                                                            },
  {Industry::IndustrialREITs,                                  {SubIndustry::IndustrialREITs}                                                             },
  {Industry::HotelAndResortREITs,                              {SubIndustry::HotelAndResortREITs}                                                         },
  {Industry::OfficeREITs,                                      {SubIndustry::OfficeREITs}                                                                 },
  {Industry::HealthCareREITs,                                  {SubIndustry::HealthCareREITs}                                                             },
  {Industry::ResidentialREITs,
   {
     SubIndustry::MultiFamilyResidentialREITs,
     SubIndustry::SingleFamilyResidentialREITs,
   }                                                                                                                                                      },
  {Industry::RetailREITs,                                      {SubIndustry::RetailREITs}                                                                 },
  {Industry::SpecializedREITs,
   {
     SubIndustry::OtherSpecializedREITs,
     SubIndustry::SelfStorageREITs,
     SubIndustry::TelecomTowerREITs,
     SubIndustry::TimberREITs,
     SubIndustry::DataCenterREITs,
   }                                                                                                                                                      },
  {Industry::RealEstateManagementAndDevelopment,
   {
     SubIndustry::DiversifiedRealEstateActivities,
     SubIndustry::RealEstateOperatingCompanies,
     SubIndustry::RealEstateDevelopment,
     SubIndustry::RealEstateServices,
   }                                                                                                                                                      }
};

std::vector<SubIndustry> inline Industry_to_SubIndustry(Industry& x) { return IndustryToSubIndustries[x]; }

static std::unordered_map<SubIndustry, Industry> SubIndustryToIndustry = {
  {SubIndustry::OilAndGasDrilling,                                    Industry::EnergyEquipmentAndServices                      },
  {SubIndustry::OilAndGasEquipmentAndServices,                        Industry::EnergyEquipmentAndServices                      },
  {SubIndustry::IntegratedOilAndGas,                                  Industry::OilGasAndConsumeableFules                       },
  {SubIndustry::OilAndGasExplorationAndProduction,                    Industry::OilGasAndConsumeableFules                       },
  {SubIndustry::OilAndGasRefiningAndMarketing,                        Industry::OilGasAndConsumeableFules                       },
  {SubIndustry::OilAndGasStorageAndTransportation,                    Industry::OilGasAndConsumeableFules                       },
  {SubIndustry::CoalAndConsumeableFuels,                              Industry::OilGasAndConsumeableFules                       },
  {SubIndustry::CommodityChemicals,                                   Industry::Chemicals                                       },
  {SubIndustry::DiversifiedChemicals,                                 Industry::Chemicals                                       },
  {SubIndustry::FertilizersAndAgriculturalChemicals,                  Industry::Chemicals                                       },
  {SubIndustry::IndustrialGases,                                      Industry::Chemicals                                       },
  {SubIndustry::SpecialtyChemicals,                                   Industry::Chemicals                                       },
  {SubIndustry::ConstructionMaterials,                                Industry::ConstructionMaterials                           },
  {SubIndustry::MetalGlassAndPlasticContainers,                       Industry::ContainersAndPackaging                          },
  {SubIndustry::PaperAndPlasticPackagingProductsAndMaterials,         Industry::ContainersAndPackaging                          },
  {SubIndustry::Aluminum,                                             Industry::MetalsAndMining                                 },
  {SubIndustry::DiversifiedMetalsAndMining,                           Industry::MetalsAndMining                                 },
  {SubIndustry::Copper,                                               Industry::MetalsAndMining                                 },
  {SubIndustry::Gold,                                                 Industry::MetalsAndMining                                 },
  {SubIndustry::PreciousMetalsAndMinerals,                            Industry::MetalsAndMining                                 },
  {SubIndustry::Silver,                                               Industry::MetalsAndMining                                 },
  {SubIndustry::Steel,                                                Industry::MetalsAndMining                                 },
  {SubIndustry::ForestProducts,                                       Industry::PaperAndForestProducts                          },
  {SubIndustry::PaperProducts,                                        Industry::PaperAndForestProducts                          },
  {SubIndustry::AerospaceAndDefense,                                  Industry::AerospaceAndDefense                             },
  {SubIndustry::BuildingProducts,                                     Industry::BuildingProducts                                },
  {SubIndustry::ConstructionAndEngineering,                           Industry::ConstructionAndEngineering                      },
  {SubIndustry::ElectricalComponentsAndEquipment,                     Industry::ElectricalEquipment                             },
  {SubIndustry::HeavyElectricalEquipment,                             Industry::ElectricalEquipment                             },
  {SubIndustry::IndustrialConglomerates,                              Industry::IndustrialConglomerates                         },
  {SubIndustry::ConstructionMachineryAndHeavyTransportationEquipment, Industry::Machinery                                       },
  {SubIndustry::AgriculturalAndFarmMachinery,                         Industry::Machinery                                       },
  {SubIndustry::IndustrialMachinerySuppliesAndComponents,             Industry::Machinery                                       },
  {SubIndustry::TradingCompaniesAndDistributors,                      Industry::TradingCompaniesAndDistributors                 },
  {SubIndustry::CommercialPrinting,                                   Industry::CommercialServicesAndSupplies                   },
  {SubIndustry::EnvironmentalAndFacilitiesServices,                   Industry::CommercialServicesAndSupplies                   },
  {SubIndustry::OfficeServicesAndSupplies,                            Industry::CommercialServicesAndSupplies                   },
  {SubIndustry::DiviersifiedSupportServices,                          Industry::CommercialServicesAndSupplies                   },
  {SubIndustry::SecurityAndAlarmServices,                             Industry::CommercialServicesAndSupplies                   },
  {SubIndustry::HumanResourcesAndEmploymentServices,                  Industry::ProfessionalServices                            },
  {SubIndustry::ResearchAndConsultingServices,                        Industry::ProfessionalServices                            },
  {SubIndustry::DataProcessingAndOutsourcedServices,                  Industry::ProfessionalServices                            },
  {SubIndustry::AirFreightAndLogistics,                               Industry::AirFreightAndLogistics                          },
  {SubIndustry::PassengerAirlines,                                    Industry::PassengerAirlines                               },
  {SubIndustry::MarineTransportation,                                 Industry::MarineTransportation                            },
  {SubIndustry::RailTransportation,                                   Industry::GroundTransportation                            },
  {SubIndustry::CargoGroundTransportation,                            Industry::GroundTransportation                            },
  {SubIndustry::PassengerGroundTransportation,                        Industry::GroundTransportation                            },
  {SubIndustry::AirportServices,                                      Industry::TransportationInfrastructure                    },
  {SubIndustry::HighwaysAndRailtracks,                                Industry::TransportationInfrastructure                    },
  {SubIndustry::MarinePortsAndServices,                               Industry::TransportationInfrastructure                    },
  {SubIndustry::AutomotivePartsAndEquipment,                          Industry::AutomobileComponents                            },
  {SubIndustry::TiresAndRubber,                                       Industry::AutomobileComponents                            },
  {SubIndustry::AutomobileManufacturers,                              Industry::Automobiles                                     },
  {SubIndustry::MotorcycleManufacturers,                              Industry::Automobiles                                     },
  {SubIndustry::ConsumerElectronics,                                  Industry::HouseholdDurables                               },
  {SubIndustry::HomeFurnishings,                                      Industry::HouseholdDurables                               },
  {SubIndustry::Homebuilding,                                         Industry::HouseholdDurables                               },
  {SubIndustry::HouseholdAppliances,                                  Industry::HouseholdDurables                               },
  {SubIndustry::HousewaresAndSpecialties,                             Industry::HouseholdDurables                               },
  {SubIndustry::LeisureProducts,                                      Industry::LeisureProducts                                 },
  {SubIndustry::ApparelAccessoriesAndLuxuryGoods,                     Industry::TextilesApparelAndLuxuryGoods                   },
  {SubIndustry::Footwear,                                             Industry::TextilesApparelAndLuxuryGoods                   },
  {SubIndustry::Textiles,                                             Industry::TextilesApparelAndLuxuryGoods                   },
  {SubIndustry::CasinosAndGaming,                                     Industry::HotelsRestaurantsAndLeisure                     },
  {SubIndustry::HotelsResortsAndCruiseLines,                          Industry::HotelsRestaurantsAndLeisure                     },
  {SubIndustry::LeisureFacilities,                                    Industry::HotelsRestaurantsAndLeisure                     },
  {SubIndustry::Restaurants,                                          Industry::HotelsRestaurantsAndLeisure                     },
  {SubIndustry::EducationServices,                                    Industry::DiversifiedConsumerServices                     },
  {SubIndustry::SpecializedConsumerServices,                          Industry::DiversifiedConsumerServices                     },
  {SubIndustry::Distributors,                                         Industry::Distributors                                    },
  {SubIndustry::BroadlineRetail,                                      Industry::BroadlineRetail                                 },
  {SubIndustry::ApparelRetail,                                        Industry::SpecialtyRetail                                 },
  {SubIndustry::ComputerAndElectronicsretail,                         Industry::SpecialtyRetail                                 },
  {SubIndustry::HomeImprovementRetail,                                Industry::SpecialtyRetail                                 },
  {SubIndustry::OtherSpecialtyRetail,                                 Industry::SpecialtyRetail                                 },
  {SubIndustry::AutomotiveRetail,                                     Industry::SpecialtyRetail                                 },
  {SubIndustry::HomefurnishingRetail,                                 Industry::SpecialtyRetail                                 },
  {SubIndustry::DrugRetail,                                           Industry::ConsumerStaplesDistributionAndRetail            },
  {SubIndustry::FoodDistributors,                                     Industry::ConsumerStaplesDistributionAndRetail            },
  {SubIndustry::FoodRetail,                                           Industry::ConsumerStaplesDistributionAndRetail            },
  {SubIndustry::ConsumerStaplesMerchandiseRetail,                     Industry::ConsumerStaplesDistributionAndRetail            },
  {SubIndustry::Brewers,                                              Industry::Beverages                                       },
  {SubIndustry::DistillersAndVintners,                                Industry::Beverages                                       },
  {SubIndustry::SoftDrinksAndNonAlcoholicBeverages,                   Industry::Beverages                                       },
  {SubIndustry::AgriculturalProductsAndServices,                      Industry::FoodProducts                                    },
  {SubIndustry::PackagedFoodsAndMeats,                                Industry::FoodProducts                                    },
  {SubIndustry::Tobacco,                                              Industry::Tobacco                                         },
  {SubIndustry::HouseholdProducts,                                    Industry::HouseholdProducts                               },
  {SubIndustry::PersonalCareProducts,                                 Industry::PersonalCareProducts                            },
  {SubIndustry::HealthCareEquipment,                                  Industry::HealthCareEquipmentAndSupplies                  },
  {SubIndustry::HealthCareSupplies,                                   Industry::HealthCareEquipmentAndSupplies                  },
  {SubIndustry::HealthCareDistributors,                               Industry::HealthCareProvidersAndServices                  },
  {SubIndustry::HealthCareServices,                                   Industry::HealthCareProvidersAndServices                  },
  {SubIndustry::HealthCareFacilities,                                 Industry::HealthCareProvidersAndServices                  },
  {SubIndustry::ManagedHealthCare,                                    Industry::HealthCareProvidersAndServices                  },
  {SubIndustry::HealthCareTechnology,                                 Industry::HealthCareTechnology                            },
  {SubIndustry::Biotechnology,                                        Industry::Biotechnology                                   },
  {SubIndustry::Pharmaceuticals,                                      Industry::Pharmaceuticals                                 },
  {SubIndustry::LifeSciencesToolsAndServices,                         Industry::LifeSciencesToolsAndServices                    },
  {SubIndustry::DiversifiedBanks,                                     Industry::Banks                                           },
  {SubIndustry::RegionalBanks,                                        Industry::Banks                                           },
  {SubIndustry::DiversifiedFinancialServices,                         Industry::FinancialServices                               },
  {SubIndustry::MultiSectorHoldings,                                  Industry::FinancialServices                               },
  {SubIndustry::SpecializedFinance,                                   Industry::FinancialServices                               },
  {SubIndustry::CommercialAndResidentialMortgageFinance,              Industry::FinancialServices                               },
  {SubIndustry::TransactionAndPaymentProcessingServices,              Industry::FinancialServices                               },
  {SubIndustry::ConsumerFinance,                                      Industry::ConsumerFinance                                 },
  {SubIndustry::AssetManagementAndCustodyBanks,                       Industry::CapitalMarkets                                  },
  {SubIndustry::InvestmentBankingAndBrokerage,                        Industry::CapitalMarkets                                  },
  {SubIndustry::DiversifiedCapitalMarkets,                            Industry::CapitalMarkets                                  },
  {SubIndustry::FinancialExchangesAndData,                            Industry::CapitalMarkets                                  },
  {SubIndustry::MortgageRealEstateInvestmentTrusts,                   Industry::MortgageRealEstateInvestmentTrusts              },
  {SubIndustry::InsuranceBrokers,                                     Industry::Insurance                                       },
  {SubIndustry::LifeAndHealthInsurance,                               Industry::Insurance                                       },
  {SubIndustry::MultilineInsurance,                                   Industry::Insurance                                       },
  {SubIndustry::PropertyAndCasualtyInsurance,                         Industry::Insurance                                       },
  {SubIndustry::Reinsurance,                                          Industry::Insurance                                       },
  {SubIndustry::ITConsultingAndOtherServices,                         Industry::ITServices                                      },
  {SubIndustry::InternetServicesAndInfrastructure,                    Industry::ITServices                                      },
  {SubIndustry::ApplicationSoftware,                                  Industry::Software                                        },
  {SubIndustry::SystemsSoftware,                                      Industry::Software                                        },
  {SubIndustry::CommunicationsEquipment,                              Industry::CommunicationsEquipment                         },
  {SubIndustry::TechnologyHardwareStorageAndPeripherals,              Industry::TechnologyHardwareStorageAndPeripherals         },
  {SubIndustry::ElectronicEquipmentAndInstruments,                    Industry::ElectronicEquipmentInstrumentsAndComponents     },
  {SubIndustry::ElectronicComponents,                                 Industry::ElectronicEquipmentInstrumentsAndComponents     },
  {SubIndustry::ElectronicManufacturingServices,                      Industry::ElectronicEquipmentInstrumentsAndComponents     },
  {SubIndustry::TechnologyDistributors,                               Industry::ElectronicEquipmentInstrumentsAndComponents     },
  {SubIndustry::SemiconductorMaterialsAndEquipment,                   Industry::SemiconductorsAndSemiconductorEquipment         },
  {SubIndustry::Semiconductors,                                       Industry::SemiconductorsAndSemiconductorEquipment         },
  {SubIndustry::AlternativeCarriers,                                  Industry::DiversifiedTelecommunicationServices            },
  {SubIndustry::IntegratedTelecommunicationServices,                  Industry::DiversifiedTelecommunicationServices            },
  {SubIndustry::WirelessTelecommunicationServices,                    Industry::WirelessTelecommunicationServices               },
  {SubIndustry::Advertising,                                          Industry::Media                                           },
  {SubIndustry::Broadcasting,                                         Industry::Media                                           },
  {SubIndustry::CableAndSatellite,                                    Industry::Media                                           },
  {SubIndustry::Publishing,                                           Industry::Media                                           },
  {SubIndustry::MoviesAndEntertainment,                               Industry::Entertainment                                   },
  {SubIndustry::InteractiveHomeEntertainment,                         Industry::Entertainment                                   },
  {SubIndustry::InteractiveMediaAndServices,                          Industry::InteractiveMediaAndServices                     },
  {SubIndustry::ElectricUtilities,                                    Industry::ElectricUtilities                               },
  {SubIndustry::GasUtilities,                                         Industry::GasUtilities                                    },
  {SubIndustry::MultiUtilities,                                       Industry::MultiUtilities                                  },
  {SubIndustry::WaterUtilities,                                       Industry::WaterUtilities                                  },
  {SubIndustry::IndependentPowerProducersAndEnergyTraders,            Industry::IndependentPowerAndRenewableElectricityProducers},
  {SubIndustry::RenewableElectricity,                                 Industry::IndependentPowerAndRenewableElectricityProducers},
  {SubIndustry::DiversifiedREITs,                                     Industry::DiversifiedREITs                                },
  {SubIndustry::IndustrialREITs,                                      Industry::IndustrialREITs                                 },
  {SubIndustry::HotelAndResortREITs,                                  Industry::HotelAndResortREITs                             },
  {SubIndustry::OfficeREITs,                                          Industry::OfficeREITs                                     },
  {SubIndustry::HealthCareREITs,                                      Industry::HealthCareREITs                                 },
  {SubIndustry::MultiFamilyResidentialREITs,                          Industry::ResidentialREITs                                },
  {SubIndustry::SingleFamilyResidentialREITs,                         Industry::ResidentialREITs                                },
  {SubIndustry::RetailREITs,                                          Industry::RetailREITs                                     },
  {SubIndustry::OtherSpecializedREITs,                                Industry::SpecializedREITs                                },
  {SubIndustry::SelfStorageREITs,                                     Industry::SpecializedREITs                                },
  {SubIndustry::TelecomTowerREITs,                                    Industry::SpecializedREITs                                },
  {SubIndustry::TimberREITs,                                          Industry::SpecializedREITs                                },
  {SubIndustry::DataCenterREITs,                                      Industry::SpecializedREITs                                },
  {SubIndustry::DiversifiedRealEstateActivities,                      Industry::RealEstateManagementAndDevelopment              },
  {SubIndustry::RealEstateOperatingCompanies,                         Industry::RealEstateManagementAndDevelopment              },
  {SubIndustry::RealEstateDevelopment,                                Industry::RealEstateManagementAndDevelopment              },
  {SubIndustry::RealEstateServices,                                   Industry::RealEstateManagementAndDevelopment              },
};

Industry inline SubIndustry_to_Industry(SubIndustry& x) { return SubIndustryToIndustry[x]; }

}
