"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("aws-cdk-lib");
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    let cwRole;
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    const _apiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
    _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    let cwRole;
    // Configure API access logging
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    const newProps = utils_1.consolidateProps(baseProps, params.awsIntegrationProps);
    apiGatewayIntegration = new api.AwsIntegration(newProps);
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    const overridenProps = utils_1.consolidateProps(defaultMethodOptions, params.methodOptions);
    apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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