"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const cdk = require("aws-cdk-lib");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, httpSecurityHeaders, cloudfrontFunction);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders = true, originPath, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, httpSecurityHeaders, originPath, cloudfrontFunction);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    let originRequestPolicy;
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, httpSecurityHeaders, cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.customHeaders, cloudfrontFunction);
    let cfprops;
    cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, cloudfrontFunction];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
function getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps) {
    const isLoggingDisabled = (cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.enableLogging) === false;
    const userSuppliedLogBucket = cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.logBucket;
    if (userSuppliedLogBucket && cloudFrontLoggingBucketProps) {
        throw Error('Either cloudFrontDistributionProps.logBucket or cloudFrontLoggingBucketProps can be set.');
    }
    return isLoggingDisabled
        ? undefined
        : userSuppliedLogBucket !== null && userSuppliedLogBucket !== void 0 ? userSuppliedLogBucket : s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket', cloudFrontLoggingBucketProps ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), cloudFrontLoggingBucketProps) : s3_bucket_defaults_1.DefaultS3Props());
}
function getCloudfrontFunction(httpSecurityHeaders, scope) {
    return httpSecurityHeaders ? defaultCloudfrontFunction(scope) : undefined;
}
//# sourceMappingURL=data:application/json;base64,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