"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    var _a, _b;
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            const levelOneFunction = props.existingLambdaObj.node.defaultChild;
            if ((_a = props.lambdaFunctionProps) === null || _a === void 0 ? void 0 : _a.securityGroups) {
                let ctr = 20;
                (_b = props.lambdaFunctionProps) === null || _b === void 0 ? void 0 : _b.securityGroups.forEach(sg => {
                    // TODO: Discuss with someone why I can't get R/O access to VpcConfigSecurityGroupIds
                    levelOneFunction.addOverride(`Properties.VpcConfig.SecurityGroupIds.${ctr++}`, sg.securityGroupId);
                });
            }
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        }, true);
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    utils_1.addCfnSuppressRules(lambdafunction, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        utils_1.addCfnSuppressRules(cfnLambdafunctionDefPolicy, [
            {
                id: 'W12',
                reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
            }
        ]);
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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