"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const ec2 = require("aws-cdk-lib/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role, props) {
    var _a, _b;
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        _role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    _role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = ((_a = props.endpointConfigProps) === null || _a === void 0 ? void 0 : _a.productionVariants)[0].acceleratorType;
        if (acceleratorType !== undefined) {
            _role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    _role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    _role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefaultPolicy = (_b = _role.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    utils_1.addCfnSuppressRules(roleDefaultPolicy, [
        {
            id: 'W12',
            reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
        },
        {
            id: 'W76',
            reason: 'Complex role becuase Sagemaker needs permissions to access several services',
        }
    ]);
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined &&
                ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        sagemakerNotebookProps = utils_1.consolidateProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            /** return [endpoint, endpointConfig, model] */
            return deploySagemakerEndpoint(scope, props);
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return [props.existingSagemakerEndpointObj];
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependsOn(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependsOn(endpointConfig);
        return [endpoint, endpointConfig, model];
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        utils_1.addCfnSuppressRules(cfnSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig), modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Overwrite default EndpointConfig properties
    finalEndpointConfigProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId), endpointConfigProps);
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Overwrite default Endpoint properties
    finalEndpointProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName), endpointProps);
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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