"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("aws-cdk-lib");
const smDefaults = require("./step-function-defaults");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    var _a, _b;
    let logGroup;
    let _smProps;
    // If they sent a logGroup in stateMachineProps
    if ((_a = stateMachineProps.logs) === null || _a === void 0 ? void 0 : _a.destination) {
        logGroup = (_b = stateMachineProps.logs) === null || _b === void 0 ? void 0 : _b.destination;
        _smProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!(consolidatedLogGroupProps === null || consolidatedLogGroupProps === void 0 ? void 0 : consolidatedLogGroupProps.logGroupName)) {
            const logGroupPrefix = '/aws/vendedlogs/states/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    utils_1.addCfnSuppressRules(cfnDefaultPolicy, [
        {
            id: 'W12',
            reason: `The 'LogDelivery' actions do not support resource-level authorizations`
        }
    ]);
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted({
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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