"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.consolidateProps = exports.addCfnSuppressRules = exports.generateResourceName = exports.printWarning = exports.overrideProps = void 0;
const deepmerge = require("deepmerge");
const override_warning_service_1 = require("./override-warning-service");
const log = require("npmlog");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
function isObject(val) {
    return val != null && typeof val === 'object'
        && Object.prototype.toString.call(val) === '[object Object]';
}
function isPlainObject(o) {
    if (Array.isArray(o) === true) {
        return true;
    }
    if (isObject(o) === false) {
        return false;
    }
    // If has modified constructor
    const ctor = o.constructor;
    if (typeof ctor !== 'function') {
        return false;
    }
    // If has modified prototype
    const prot = ctor.prototype;
    if (isObject(prot) === false) {
        return false;
    }
    // If constructor does not have an Object-specific method
    if (prot.hasOwnProperty('isPrototypeOf') === false) {
        return false;
    }
    // Most likely a plain Object
    return true;
}
function combineMerge(target, source) {
    return target.concat(source);
}
function overwriteMerge(target, source) {
    target = source;
    return target;
}
function overrideProps(DefaultProps, userProps, concatArray = false) {
    // Notify the user via console output if defaults are overridden
    const overrideWarningsEnabled = (process.env.overrideWarningsEnabled !== 'false');
    if (overrideWarningsEnabled) {
        override_warning_service_1.flagOverriddenDefaults(DefaultProps, userProps);
    }
    // Override the sensible defaults with user provided props
    if (concatArray) {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: combineMerge,
            isMergeableObject: isPlainObject
        });
    }
    else {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: overwriteMerge,
            isMergeableObject: isPlainObject
        });
    }
}
exports.overrideProps = overrideProps;
function printWarning(message) {
    // Style the log output
    log.prefixStyle.bold = true;
    log.prefixStyle.fg = 'red';
    log.enableColor();
    log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', message);
}
exports.printWarning = printWarning;
/**
 * @summary Creates a resource name in the style of the CDK (string+hash)
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if neccessary) + a hash of the full concatenated parts
 *
 * This is based upon this discussion - https://github.com/aws/aws-cdk/issues/1424
 */
function generateResourceName(parts, maxLength) {
    const hashLength = 12;
    const maxPartLength = Math.floor((maxLength - hashLength) / parts.length);
    const sha256 = crypto.createHash("sha256");
    let finalName = '';
    parts.forEach((part) => {
        sha256.update(part);
        finalName += removeNonAlphanumeric(part.slice(0, maxPartLength));
    });
    const hash = sha256.digest("hex").slice(0, hashLength);
    finalName += hash;
    return finalName.toLowerCase();
}
exports.generateResourceName = generateResourceName;
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    var _a, _b, _c;
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if ((_b = (_a = resource.cfnOptions.metadata) === null || _a === void 0 ? void 0 : _a.cfn_nag) === null || _b === void 0 ? void 0 : _b.rules_to_suppress) {
        (_c = resource.cfnOptions.metadata) === null || _c === void 0 ? void 0 : _c.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules
        });
    }
}
exports.addCfnSuppressRules = addCfnSuppressRules;
/**
 * Creates the props to be used to instantiate a CDK L2 construct within a Solutions Construct
 *
 * @param defaultProps The default props to be used by the construct
 * @param clientProps Optional properties passed in from the client in the props object
 * @param constructProps Optional properties required by the construct for the construct to work (override any other values)
 * @returns The properties to use - all values prioritized:
 *  1) constructProps value
 *  2) clientProps value
 *  3) defaultProps value
 */
function consolidateProps(defaultProps, clientProps, constructProps, concatArray = false) {
    let result = defaultProps;
    if (clientProps) {
        result = overrideProps(result, clientProps, concatArray);
    }
    if (constructProps) {
        result = overrideProps(result, constructProps, concatArray);
    }
    return result;
}
exports.consolidateProps = consolidateProps;
//# sourceMappingURL=data:application/json;base64,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