"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("..");
const _fieldSchema = [{
        name: "id",
        type: "int",
        comment: "Identifier for the record"
    }, {
        name: "name",
        type: "string",
        comment: "The name of the record"
    }, {
        name: "type",
        type: "string",
        comment: "The type of the record"
    }, {
        name: "numericvalue",
        type: "int",
        comment: "Some value associated with the record"
    }];
// --------------------------------------------------------------
// Create database by passing props
// --------------------------------------------------------------
test('create default CfnTable with default props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.DefaultGlueTable(stack, {
        catalogId: 'fakecatalogfortest',
        databaseName: 'fakedatabase',
        tableInput: {
            parameters: {
                classification: 'json'
            },
            storageDescriptor: {
                parameters: {
                    endpointUrl: `https://kinesis.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`,
                    streamName: 'testStream',
                    typeOfData: 'kinesis'
                }
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Type: "AWS::Glue::Table",
        Properties: {
            CatalogId: "fakecatalogfortest",
            DatabaseName: "fakedatabase",
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    }
                }
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Create database by passing no props to database (default database)
// --------------------------------------------------------------
test('Create table', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.createGlueTable(stack, defaults.createGlueDatabase(stack), undefined, _fieldSchema, 'kinesis', {
        STREAM_NAME: 'testStream'
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Database', {
        Type: "AWS::Glue::Database",
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseInput: {
                Description: "An AWS Glue database generated by AWS Solutions Construct"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "GlueDatabase"
            },
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Comment: "Identifier for the record",
                            Name: "id",
                            Type: "int"
                        },
                        {
                            Comment: "The name of the record",
                            Name: "name",
                            Type: "string"
                        },
                        {
                            Comment: "The type of the record",
                            Name: "type",
                            Type: "string"
                        },
                        {
                            Comment: "Some value associated with the record",
                            Name: "numericvalue",
                            Type: "int"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: "testStream",
                    NumberOfBuckets: -1,
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    },
                    SerdeInfo: {
                        Parameters: {
                            paths: "id,name,type,numericvalue"
                        },
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    }
                },
                TableType: "EXTERNAL_TABLE"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an unsupported source type, it should throw an error
// --------------------------------------------------------------
test('error condition', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const _database = defaults.DefaultGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
        defaults.DefaultGlueTable(_database, defaults.DefaultGlueTableProps(_database, _fieldSchema, 'SomeSource', { STREAM_NAME: 'somefakestream' }));
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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