"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const ec2 = require("aws-cdk-lib/aws-ec2");
const defaults = require("../");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with VPC
// --------------------------------------------------------------
test('Test deployment with VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    let sagemaker;
    let vpc;
    let sg;
    // Build Sagemaker Notebook Instance
    [sagemaker, vpc, sg] = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.privateSubnets.length).toEqual(2);
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.publicSubnets.length).toEqual(2);
    expect(sagemaker.instanceType).toEqual('ml.t2.medium');
    expect(sg).toBeInstanceOf(ec2.SecurityGroup);
});
// --------------------------------------------------------------
// Test deployment in existing VPC
// --------------------------------------------------------------
test('Test deployment w/ existing VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef'],
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        SecurityGroupIds: ['sg-deadbeef'],
        SubnetId: 'subnet-deadbeef',
    });
});
// --------------------------------------------------------------
// Test deployment with override
// --------------------------------------------------------------
test('Test deployment w/ override', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    const key = new kms.Key(stack, 'MyEncryptionKey');
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        sagemakerNotebookProps: {
            instanceType: 'ml.c4.2xlarge',
            kmsKeyId: key.keyArn,
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        InstanceType: 'ml.c4.2xlarge',
        KmsKeyId: {
            'Fn::GetAtt': ['MyEncryptionKeyD795679F', 'Arn'],
        },
    });
});
// --------------------------------------------------------------
// Test exception
// --------------------------------------------------------------
test('Test exception', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    expect(() => {
        // Build Sagemaker Notebook Instance
        defaults.buildSagemakerNotebook(stack, {
            role: sagemakerRole,
            deployInsideVpc: true,
            sagemakerNotebookProps: {
                subnetId: 'subnet-deadbeef',
            },
        });
    }).toThrowError();
});
// ---------------------------------------------------------------
// Test exception for not providing primaryContainer in modelProps
// ---------------------------------------------------------------
test('Test exception for not providing primaryContainer in modelProps', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, {
            modelProps: {},
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps or existingSagemkaerObj
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps or existingSagemkaerObj', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -----------------------------------------------------------------------------------------
// Test exception for not providing private or isolated subnets in an existing vpc
// -----------------------------------------------------------------------------------------
test('Test exception for not providing private or isolated subnets in an existing vpc', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // create a VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        userVpcProps: {
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'public',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
            ],
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, {
            modelProps: {
                primaryContainer: {
                    image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                    modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
                },
            },
            vpc,
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLWhlbHBlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2FnZW1ha2VyLWhlbHBlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCw2Q0FBb0M7QUFDcEMsMkNBQTJDO0FBQzNDLDJDQUEyQztBQUMzQywyQ0FBMkM7QUFDM0MsZ0NBQWdDO0FBQ2hDLGdDQUE4QjtBQUU5QixpRUFBaUU7QUFDakUsMkJBQTJCO0FBQzNCLGlFQUFpRTtBQUNqRSxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO0lBQ3BDLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUN6RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7S0FDL0QsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxTQUFTLENBQUM7SUFDZCxJQUFJLEdBQUcsQ0FBQztJQUNSLElBQUksRUFBRSxDQUFDO0lBRVAsb0NBQW9DO0lBQ3BDLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQzVELElBQUksRUFBRSxhQUFhO0tBQ3BCLENBQUMsQ0FBQztJQUNILFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUMsTUFBTSxDQUFDLEdBQUcsYUFBSCxHQUFHLHVCQUFILEdBQUcsQ0FBRSxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzdDLE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3ZELE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQy9DLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGtDQUFrQztBQUNsQyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMzQyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUNILG9DQUFvQztJQUNwQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ3JDLElBQUksRUFBRSxhQUFhO1FBQ25CLGVBQWUsRUFBRSxJQUFJO1FBQ3JCLHNCQUFzQixFQUFFO1lBQ3RCLFFBQVEsRUFBRSxpQkFBaUI7WUFDM0IsZ0JBQWdCLEVBQUUsQ0FBQyxhQUFhLENBQUM7U0FDbEM7S0FDRixDQUFDLENBQUM7SUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtDQUFrQyxFQUFFO1FBQy9ELG9CQUFvQixFQUFFLFVBQVU7UUFDaEMsZ0JBQWdCLEVBQUUsQ0FBQyxhQUFhLENBQUM7UUFDakMsUUFBUSxFQUFFLGlCQUFpQjtLQUM1QixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSxnQ0FBZ0M7QUFDaEMsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQ3pELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztLQUMvRCxDQUFDLENBQUM7SUFDSCxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDbEQsb0NBQW9DO0lBQ3BDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDckMsSUFBSSxFQUFFLGFBQWE7UUFDbkIsc0JBQXNCLEVBQUU7WUFDdEIsWUFBWSxFQUFFLGVBQWU7WUFDN0IsUUFBUSxFQUFFLEdBQUcsQ0FBQyxNQUFNO1NBQ3JCO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQ0FBa0MsRUFBRTtRQUMvRCxvQkFBb0IsRUFBRSxVQUFVO1FBQ2hDLFlBQVksRUFBRSxlQUFlO1FBQzdCLFFBQVEsRUFBRTtZQUNSLFlBQVksRUFBRSxDQUFDLHlCQUF5QixFQUFFLEtBQUssQ0FBQztTQUNqRDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGlCQUFpQjtBQUNqQixpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEdBQUcsRUFBRTtJQUMxQixRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixvQ0FBb0M7UUFDcEMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLEtBQUssRUFBRTtZQUNyQyxJQUFJLEVBQUUsYUFBYTtZQUNuQixlQUFlLEVBQUUsSUFBSTtZQUNyQixzQkFBc0IsRUFBRTtnQkFDdEIsUUFBUSxFQUFFLGlCQUFpQjthQUM1QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDO0FBQ3BCLENBQUMsQ0FBQyxDQUFDO0FBRUgsa0VBQWtFO0FBQ2xFLGtFQUFrRTtBQUNsRSxrRUFBa0U7QUFDbEUsSUFBSSxDQUFDLGlFQUFpRSxFQUFFLEdBQUcsRUFBRTtJQUMzRSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YscUNBQXFDO1FBQ3JDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7WUFDckMsVUFBVSxFQUFFLEVBQUU7U0FDZixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFDRixjQUFjO0lBQ2QsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDO0FBQzdCLENBQUMsQ0FBQyxDQUFDO0FBRUgsNEVBQTRFO0FBQzVFLDhDQUE4QztBQUM5Qyw0RUFBNEU7QUFDNUUsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtJQUN2RCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7UUFDbkMsZUFBZSxFQUFFLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRTtRQUNuRCxpQkFBaUIsRUFBRTtZQUNqQixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLGdCQUFnQixFQUFFLElBQUk7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixxQ0FBcUM7UUFDckMsUUFBUSxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFDbkQsQ0FBQyxDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILDRFQUE0RTtBQUM1RSxzRUFBc0U7QUFDdEUsNEVBQTRFO0FBQzVFLElBQUksQ0FBQyxxRUFBcUUsRUFBRSxHQUFHLEVBQUU7SUFDL0UsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sR0FBRyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFO1FBQ25DLGVBQWUsRUFBRSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7UUFDbkQsaUJBQWlCLEVBQUU7WUFDakIsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixnQkFBZ0IsRUFBRSxJQUFJO1NBQ3ZCO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YscUNBQXFDO1FBQ3JDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQ2xELENBQUMsQ0FBQztJQUNGLGNBQWM7SUFDZCxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDN0IsQ0FBQyxDQUFDLENBQUM7QUFFSCw0RkFBNEY7QUFDNUYsa0ZBQWtGO0FBQ2xGLDRGQUE0RjtBQUM1RixJQUFJLENBQUMsaUZBQWlGLEVBQUUsR0FBRyxFQUFFO0lBQzNGLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixlQUFlO0lBQ2YsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7UUFDbkMsZUFBZSxFQUFFLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRTtRQUNuRCxZQUFZLEVBQUU7WUFDWixXQUFXLEVBQUUsQ0FBQztZQUNkLG1CQUFtQixFQUFFO2dCQUNuQjtvQkFDRSxRQUFRLEVBQUUsRUFBRTtvQkFDWixJQUFJLEVBQUUsUUFBUTtvQkFDZCxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxNQUFNO2lCQUNsQzthQUNGO1NBQ0Y7UUFDRCxpQkFBaUIsRUFBRTtZQUNqQixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLGdCQUFnQixFQUFFLElBQUk7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixxQ0FBcUM7UUFDckMsUUFBUSxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRTtZQUN0QyxVQUFVLEVBQUU7Z0JBQ1YsZ0JBQWdCLEVBQUU7b0JBQ2hCLEtBQUssRUFBRSxrRUFBa0U7b0JBQ3pFLFlBQVksRUFBRSwwQ0FBMEM7aUJBQ3pEO2FBQ0Y7WUFDRCxHQUFHO1NBQ0osQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIyIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICcuLi8nO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgd2l0aCBWUENcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgd2l0aCBWUEMnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcbiAgbGV0IHNhZ2VtYWtlcjtcbiAgbGV0IHZwYztcbiAgbGV0IHNnO1xuXG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICBbc2FnZW1ha2VyLCB2cGMsIHNnXSA9IGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICByb2xlOiBzYWdlbWFrZXJSb2xlLFxuICB9KTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdCh2cGM/LnByaXZhdGVTdWJuZXRzLmxlbmd0aCkudG9FcXVhbCgyKTtcbiAgZXhwZWN0KHZwYz8ucHVibGljU3VibmV0cy5sZW5ndGgpLnRvRXF1YWwoMik7XG4gIGV4cGVjdChzYWdlbWFrZXIuaW5zdGFuY2VUeXBlKS50b0VxdWFsKCdtbC50Mi5tZWRpdW0nKTtcbiAgZXhwZWN0KHNnKS50b0JlSW5zdGFuY2VPZihlYzIuU2VjdXJpdHlHcm91cCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgZGVwbG95bWVudCBpbiBleGlzdGluZyBWUENcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgVlBDJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBzYWdlbWFrZXJSb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnU2FnZW1ha2VyUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgfSk7XG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICBkZWZhdWx0cy5idWlsZFNhZ2VtYWtlck5vdGVib29rKHN0YWNrLCB7XG4gICAgcm9sZTogc2FnZW1ha2VyUm9sZSxcbiAgICBkZXBsb3lJbnNpZGVWcGM6IHRydWUsXG4gICAgc2FnZW1ha2VyTm90ZWJvb2tQcm9wczoge1xuICAgICAgc3VibmV0SWQ6ICdzdWJuZXQtZGVhZGJlZWYnLFxuICAgICAgc2VjdXJpdHlHcm91cElkczogWydzZy1kZWFkYmVlZiddLFxuICAgIH0sXG4gIH0pO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlNhZ2VNYWtlcjo6Tm90ZWJvb2tJbnN0YW5jZScsIHtcbiAgICBEaXJlY3RJbnRlcm5ldEFjY2VzczogJ0Rpc2FibGVkJyxcbiAgICBTZWN1cml0eUdyb3VwSWRzOiBbJ3NnLWRlYWRiZWVmJ10sXG4gICAgU3VibmV0SWQ6ICdzdWJuZXQtZGVhZGJlZWYnLFxuICB9KTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHdpdGggb3ZlcnJpZGVcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gb3ZlcnJpZGUnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcbiAgY29uc3Qga2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdNeUVuY3J5cHRpb25LZXknKTtcbiAgLy8gQnVpbGQgU2FnZW1ha2VyIE5vdGVib29rIEluc3RhbmNlXG4gIGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICByb2xlOiBzYWdlbWFrZXJSb2xlLFxuICAgIHNhZ2VtYWtlck5vdGVib29rUHJvcHM6IHtcbiAgICAgIGluc3RhbmNlVHlwZTogJ21sLmM0LjJ4bGFyZ2UnLFxuICAgICAga21zS2V5SWQ6IGtleS5rZXlBcm4sXG4gICAgfSxcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6U2FnZU1ha2VyOjpOb3RlYm9va0luc3RhbmNlJywge1xuICAgIERpcmVjdEludGVybmV0QWNjZXNzOiAnRGlzYWJsZWQnLFxuICAgIEluc3RhbmNlVHlwZTogJ21sLmM0LjJ4bGFyZ2UnLFxuICAgIEttc0tleUlkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFsnTXlFbmNyeXB0aW9uS2V5RDc5NTY3OUYnLCAnQXJuJ10sXG4gICAgfSxcbiAgfSk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgZXhjZXB0aW9uXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBleGNlcHRpb24nLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcblxuICBleHBlY3QoKCkgPT4ge1xuICAgIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICAgIGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICAgIHJvbGU6IHNhZ2VtYWtlclJvbGUsXG4gICAgICBkZXBsb3lJbnNpZGVWcGM6IHRydWUsXG4gICAgICBzYWdlbWFrZXJOb3RlYm9va1Byb3BzOiB7XG4gICAgICAgIHN1Ym5ldElkOiAnc3VibmV0LWRlYWRiZWVmJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pLnRvVGhyb3dFcnJvcigpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBleGNlcHRpb24gZm9yIG5vdCBwcm92aWRpbmcgcHJpbWFyeUNvbnRhaW5lciBpbiBtb2RlbFByb3BzXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgZXhjZXB0aW9uIGZvciBub3QgcHJvdmlkaW5nIHByaW1hcnlDb250YWluZXIgaW4gbW9kZWxQcm9wcycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgLy8gQnVpbGQgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludFxuICAgIGRlZmF1bHRzLkJ1aWxkU2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHtcbiAgICAgIG1vZGVsUHJvcHM6IHt9LFxuICAgIH0pO1xuICB9O1xuICAvLyBBc3NlcnRpb24gMVxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBCdWlsZCBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50XG4gICAgZGVmYXVsdHMuZGVwbG95U2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHsgdnBjIH0pO1xuICB9O1xuICAvLyBBc3NlcnRpb24gMVxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzIG9yIGV4aXN0aW5nU2FnZW1rYWVyT2JqXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzIG9yIGV4aXN0aW5nU2FnZW1rYWVyT2JqJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBCdWlsZCBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50XG4gICAgZGVmYXVsdHMuQnVpbGRTYWdlbWFrZXJFbmRwb2ludChzdGFjaywgeyB2cGMgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvbiAxXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBwcml2YXRlIG9yIGlzb2xhdGVkIHN1Ym5ldHMgaW4gYW4gZXhpc3RpbmcgdnBjXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBleGNlcHRpb24gZm9yIG5vdCBwcm92aWRpbmcgcHJpdmF0ZSBvciBpc29sYXRlZCBzdWJuZXRzIGluIGFuIGV4aXN0aW5nIHZwYycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAvLyBjcmVhdGUgYSBWUENcbiAgY29uc3QgdnBjID0gZGVmYXVsdHMuYnVpbGRWcGMoc3RhY2ssIHtcbiAgICBkZWZhdWx0VnBjUHJvcHM6IGRlZmF1bHRzLkRlZmF1bHRJc29sYXRlZFZwY1Byb3BzKCksXG4gICAgdXNlclZwY1Byb3BzOiB7XG4gICAgICBuYXRHYXRld2F5czogMCxcbiAgICAgIHN1Ym5ldENvbmZpZ3VyYXRpb246IFtcbiAgICAgICAge1xuICAgICAgICAgIGNpZHJNYXNrOiAxOCxcbiAgICAgICAgICBuYW1lOiAncHVibGljJyxcbiAgICAgICAgICBzdWJuZXRUeXBlOiBlYzIuU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgY29uc3RydWN0VnBjUHJvcHM6IHtcbiAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IHRydWUsXG4gICAgfSxcbiAgfSk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIC8vIEJ1aWxkIFNhZ2VtYWtlciBJbmZlcmVuY2UgRW5kcG9pbnRcbiAgICBkZWZhdWx0cy5kZXBsb3lTYWdlbWFrZXJFbmRwb2ludChzdGFjaywge1xuICAgICAgbW9kZWxQcm9wczoge1xuICAgICAgICBwcmltYXJ5Q29udGFpbmVyOiB7XG4gICAgICAgICAgaW1hZ2U6ICc8QWNjb3VudElkPi5ka3IuZWNyLjxyZWdpb24+LmFtYXpvbmF3cy5jb20vbGluZWFyLWxlYXJuZXI6bGF0ZXN0JyxcbiAgICAgICAgICBtb2RlbERhdGFVcmw6ICdzMzovLzxidWNrZXQtbmFtZT4vPHByZWZpeD4vbW9kZWwudGFyLmd6JyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICB2cGMsXG4gICAgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvbiAxXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7XG4iXX0=