"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTestCache = exports.suppressAutoDeleteHandlerWarnings = exports.getFakeCertificate = exports.getTestVpc = exports.generateIntegStackName = exports.CreateScrapBucket = exports.fakeEcrRepoArn = void 0;
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
const utils_1 = require("../lib/utils");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const path = require("path");
const cache = require("aws-cdk-lib/aws-elasticache");
const ec2 = require("aws-cdk-lib/aws-ec2");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
exports.fakeEcrRepoArn = 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo';
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
// Helper Functions
function getTestVpc(stack, publicFacing = true) {
    return vpc_helper_1.buildVpc(stack, {
        defaultVpcProps: publicFacing ?
            vpc_defaults_1.DefaultPublicPrivateVpcProps() :
            vpc_defaults_1.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
}
exports.getTestVpc = getTestVpc;
function getFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
exports.getFakeCertificate = getFakeCertificate;
function suppressAutoDeleteHandlerWarnings(stack) {
    aws_cdk_lib_1.Stack.of(stack).node.children.forEach(child => {
        if (child.node.id === 'Custom::S3AutoDeleteObjectsCustomResourceProvider') {
            const handlerFunction = child.node.findChild('Handler');
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W58", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W89", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W92", reason: "CDK generated custom resource" }]);
        }
    });
}
exports.suppressAutoDeleteHandlerWarnings = suppressAutoDeleteHandlerWarnings;
function CreateTestCache(scope, id, vpc, port) {
    const cachePort = port !== null && port !== void 0 ? port : elasticache_defaults_1.GetDefaultCachePort();
    // Create the subnet group from all the isolated subnets in the VPC
    const subnetGroup = elasticache_helper_1.createCacheSubnetGroup(scope, vpc, id);
    const emptySG = new ec2.SecurityGroup(scope, `${id}-cachesg`, {
        vpc,
        allowAllOutbound: true,
    });
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W40", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W5", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W36", reason: "Test Resource" }]);
    const cacheProps = {
        clusterName: `${id}-cdk-cluster`,
        cacheNodeType: "cache.t3.medium",
        engine: "memcached",
        numCacheNodes: 2,
        port: cachePort,
        azMode: "cross-az",
        vpcSecurityGroupIds: [emptySG.securityGroupId],
        cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
    };
    const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, cacheProps);
    newCache.addDependsOn(subnetGroup);
    return newCache;
}
exports.CreateTestCache = CreateTestCache;
//# sourceMappingURL=data:application/json;base64,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