"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 */
class Aws {
    constructor() { }
}
exports.Aws = Aws;
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
Aws.PARTITION = pseudoString(AWS_PARTITION);
Aws.REGION = pseudoString(AWS_REGION);
Aws.STACK_ID = pseudoString(AWS_STACKID);
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 */
class ScopedAws {
    constructor(scope) {
        this.scope = scope;
    }
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS
        });
    }
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    get region() {
        return this.asString(AWS_REGION);
    }
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    return token_1.Token.asString({ Ref: name }, { displayHint: name });
}
//# sourceMappingURL=data:application/json;base64,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