"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
// import required to be here, otherwise causes a cycle when running the generated JavaScript
// tslint:disable-next-line:ordered-imports
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const deps_1 = require("./deps");
const cfn_reference_1 = require("./private/cfn-reference");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     * @param cfnResourceType The CloudFormation type of this resource (e.g. AWS::DynamoDB::Table)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (this.node.tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.cfnOptions.metadata = {
                [cxapi.PATH_METADATA_KEY]: this.node.path
            };
        }
    }
    /**
     * Check whether the given construct is a CfnResource
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * For example,
     * ```typescript
     * addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute'])
     * addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE')
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *   "GlobalSecondaryIndexes": [
     *     {
     *       "Projection": {
     *         "NonKeyAttributes": [ "myattribute" ]
     *         ...
     *       }
     *       ...
     *     },
     *     {
     *       "ProjectionType": "INCLUDE"
     *       ...
     *     },
     *   ]
     *   ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split('.');
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be
     * provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     */
    addDependsOn(target) {
        deps_1.addDependency(this, target, `"${this.node.path}" depends on "${target.node.path}"`);
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId
                    }, props => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties = renderedProps && (Object.values(renderedProps).find(v => !!v) ? renderedProps : undefined);
                        return deepMerge(props, this.rawOverrides);
                    })
                }
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = { ...policy };
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    get cfnProperties() {
        const props = this._cfnProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.CfnResource = CfnResource;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,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